#pragma once

#include <yandex_io/libs/metrica/db/i_events_db.h>
#include <yandex_io/libs/base/named_callback_queue.h>
#include <yandex_io/libs/logging/logging.h>

#include <chrono>

namespace quasar {
    /* Batch should implement methods:
        bool append(YandexIO::EventsDB::Event event, YandexIO::EventsDB::Environment env);
        void reset();
        bool hasUnsent();
        uint32_t size();
     */

    template <typename T>
    concept BatchContainer =
        requires(T& t, YandexIO::EventsDB::Event event, YandexIO::EventsDB::Environment env) {
        { t.hasUnsent() } -> std::same_as<bool>;
        { t.append(event, env) } -> std::same_as<bool>;
        { t.reset() } -> std::same_as<void>;
        { t.size() } -> std::same_as<std::uint32_t>;
    };

    template <BatchContainer Batch>
    class Batcher {
        class SinkImpl: public YandexIO::EventsDB::Sink {
            Batcher& owner_;

        public:
            SinkImpl(Batcher* owner)
                : owner_(*owner)
            {
            }

            void handleDbEvent(YandexIO::EventsDB::ISourceControl& ctrl, YandexIO::EventsDB::Event event, YandexIO::EventsDB::Environment env) override {
                owner_.handleDbEvent(ctrl, std::move(event), std::move(env));
            }
        };

    public:
        Batcher(std::string queueName)
            : name_(queueName)
            , queue_(std::make_shared<quasar::NamedCallbackQueue>(std::move(queueName), 100))
            , lastFlushTime_(std::chrono::steady_clock::now())
        {
        }

        virtual ~Batcher() = default;

        std::shared_ptr<YandexIO::EventsDB::Sink> getSink() {
            if (!sink_) {
                sink_ = std::make_shared<SinkImpl>(this);
            }
            return sink_;
        }

        std::shared_ptr<quasar::ICallbackQueue> getCbQueue() const {
            return queue_;
        }

    protected:
        Batch batch_;
        virtual bool sendBatch() = 0;
        virtual std::chrono::seconds getMaxBatchCollectingTime() = 0;

        const std::string name_;
        std::shared_ptr<quasar::ICallbackQueue> queue_;
        std::shared_ptr<YandexIO::EventsDB::Sink> sink_;

    private:
        void handleDbEvent(YandexIO::EventsDB::ISourceControl& ctrl, YandexIO::EventsDB::Event event, YandexIO::EventsDB::Environment env) {
            if (batch_.append(std::move(event), std::move(env)) && !batchShouldBeFlushedByTime()) {
                // init timer on first event in batch.
                // Appmetica's batch can be non empty after reset and batch can skip events eg by size
                tryToStartSendTimer();
                ctrl.readyForNext();
            } else {
                const bool hasUnsentEvent = batch_.hasUnsent();
                readyForNext_ = [&ctrl, hasUnsentEvent]() {
                    if (hasUnsentEvent) {
                        ctrl.releaseBeforeLast();
                    } else {
                        ctrl.releaseIncludingLast();
                    }
                    ctrl.readyForNext();
                };
                ++batchId_;
                flushBatch();
            }
        }

        void tryToStartSendTimer() {
            if (batchReseted_ && batch_.size()) {
                batchReseted_ = false;
                queue_->addDelayed([this, batchId = batchId_]() {
                    flushBatchByTime(batchId);
                }, getMaxBatchCollectingTime());
            }
        }

        void flushBatchByTime(unsigned batchId) {
            if (batchId == batchId_) {
                YIO_LOG_INFO(name_ << " Flushing batch by timeout");
                ++batchId_;
                flushBatch();
            }
        }

        bool batchShouldBeFlushedByTime() {
            const auto now = std::chrono::steady_clock::now();
            return now - lastFlushTime_ >= getMaxBatchCollectingTime();
        }

        void flushBatch() {
            YIO_LOG_INFO(name_ << " Attempt of flushing batch number " << (batchId_ - 1) << " of size " << batch_.size());
            if (sendBatch()) {
                batch_.reset();
                batchReseted_ = true;
                lastFlushTime_ = std::chrono::steady_clock::now();
                tryToStartSendTimer();
                if (readyForNext_) {
                    readyForNext_();
                    readyForNext_ = nullptr;
                }
            } else {
                YIO_LOG_INFO(name_ << " send failed, requeueing attempt");
                queue_->add([this]() {
                    flushBatch();
                });
            }
        }

        std::function<void()> readyForNext_;
        unsigned batchId_{0};
        bool batchReseted_{true};
        std::chrono::steady_clock::time_point lastFlushTime_;
    };

} // namespace quasar
