#include <yandex_io/libs/metrica/clickdaemon/clickdaemon_metadata.h>
#include <yandex_io/libs/json_utils/json_utils.h>

#include "default_sinks_configurer.h"

using namespace quasar;

namespace {
    Json::Value clickdaemonConfig(const std::shared_ptr<YandexIO::IDevice>& device) {
        const auto& serviceConfig = device->configuration()->getServiceConfig("metrica2d");
        return serviceConfig["consumers"]["clickdaemon"];
    }

    ClickdaemonBatcher::Config makeClickdaemonConfig(const std::shared_ptr<YandexIO::IDevice>& device) {
        ClickdaemonBatcher::Config result;
        const auto config = clickdaemonConfig(device);
        result.maxEventsInBatch = getUInt64(config, "eventsBatchSize");
        result.endpointUri = getString(config, "endpointUri");
        return result;
    }

    std::string getClickdaemonUuid(const std::shared_ptr<YandexIO::IDevice>& device) {
        const auto config = clickdaemonConfig(device);
        const auto metadata = ClickdaemonMetadata::load(getString(config, "metricaMetadataPath"));
        return metadata.UUID;
    }

    void initClickdaemonFilter(const std::shared_ptr<YandexIO::IDevice>& device, YandexIO::EventsDB::EventsFilter& dst) {
        auto config = clickdaemonConfig(device);
        dst.isWhiteList = true;
        dst.eventNames = tryGetEmplaceableStringSet<std::set<std::string>>(config, "eventWhitelist", {});
        dst.envBlackList = tryGetEmplaceableStringSet<std::set<std::string>>(config, "envKeysBlacklist", {});
    }

    void initAppmetricaFilter(const std::shared_ptr<YandexIO::IDevice>& device, YandexIO::EventsDB::EventsFilter& dst) {
        const auto& config = device->configuration()->getServiceConfig(Teleme3dEndpoint::SERVICE_NAME);
        dst.isWhiteList = false; // by default it pass all
        dst.eventNames = tryGetEmplaceableStringSet<std::set<std::string>>(config, "eventBlacklist", {});
        dst.envBlackList = tryGetEmplaceableStringSet<std::set<std::string>>(config, "envKeysBlacklist", {});
    };

} // namespace

DefaultSinksConfigurer::DefaultSinksConfigurer(std::shared_ptr<YandexIO::IDevice> device)
    : appmetricaBatcher_(device)
    , clickdaemonBatcher_(device, makeClickdaemonConfig(device), getClickdaemonUuid(device))
{
}

void DefaultSinksConfigurer::registerSinks(const std::shared_ptr<YandexIO::IDevice>& device, const std::shared_ptr<YandexIO::ITelemetryEventsDB>& eventsDb) {
    {
        YandexIO::EventsDB::EventsFilter clickdaemonFilter;
        initClickdaemonFilter(device, clickdaemonFilter);
        auto clickdaemonSrcControl = eventsDb->registerSink(0, clickdaemonBatcher_.getSink(), clickdaemonBatcher_.getCbQueue(), clickdaemonFilter);
        clickdaemonSrcControl->readyForNext();
    }

    YandexIO::EventsDB::EventsFilter appmetricaFilter;
    initAppmetricaFilter(device, appmetricaFilter);
    auto appmetricaSrcControl = eventsDb->registerSink(1, appmetricaBatcher_.getSink(), appmetricaBatcher_.getCbQueue(), appmetricaFilter);
    appmetricaBatcher_.setOnReady([appmetricaSrcControl = std::move(appmetricaSrcControl)]() {
        YIO_LOG_INFO("Appmetrica is ready to send events");
        appmetricaSrcControl->readyForNext();
    });
}

void DefaultSinksConfigurer::setTimezoneOffsetSec(int32_t offsetSec) {
    appmetricaBatcher_.setTimezoneOffsetSec(offsetSec);
}

void DefaultSinksConfigurer::setLocation(double lat, double lon) {
    appmetricaBatcher_.setLocation(lat, lon);
}

void DefaultSinksConfigurer::updateConfig(const Json::Value& /*config*/) {
    // FIXME: implement
}
