#include "teleme3d_endpoint.h"

#include <yandex_io/libs/threading/callback_queue.h>
#include <yandex_io/libs/base/named_callback_queue.h>
#include <yandex_io/libs/logging/logging.h>
#include <yandex_io/protos/quasar_proto.pb.h>
#include <yandex_io/libs/protobuf_utils/debug.h>
#include <yandex_io/libs/metrica/db/lmdb_events.h>
#include <yandex_io/libs/json_utils/json_utils.h>

#include <chrono>

using namespace quasar;
using namespace YandexIO;

using quasar::proto::QuasarMessage;

const std::string Teleme3dEndpoint::SERVICE_NAME = "metricad";

namespace {
    uint64_t getNowInSeconds() {
        const auto now = std::chrono::system_clock::now();
        auto timeInSec = std::chrono::duration_cast<std::chrono::seconds>(now.time_since_epoch()).count();
        return static_cast<uint64_t>(timeInSec);
    }

    MetricaSessionProvider makeSession(const std::shared_ptr<YandexIO::IDevice>& device) {
        const auto& serviceConfig = device->configuration()->getServiceConfig(Teleme3dEndpoint::SERVICE_NAME);
        std::string sessionIdPersistentPart = getString(serviceConfig, "metricaSessionIdPersistentPart");
        std::string sessionIdTemporaryPart = getString(serviceConfig, "metricaSessionIdTemporaryPart");
        return MetricaSessionProvider(std::move(sessionIdPersistentPart), std::move(sessionIdTemporaryPart));
    }

    struct DbConfig {
        std::string path;
        std::uint64_t sizeLimit;
        EventsDB::Config cfg;
    };

    DbConfig getDbConfig(const std::shared_ptr<YandexIO::IDevice>& device) {
        const auto& serviceConfig = device->configuration()->getServiceConfig(Teleme3dEndpoint::SERVICE_NAME);
        const auto& teleme3dConfig = serviceConfig["teleme3d"];
        DbConfig result;
        result.path = getString(teleme3dConfig, "dbPath");
        result.sizeLimit = tryGetFloat(teleme3dConfig, "dbSizeLimitMB", 2.0) * 1024 * 1024;
        std::map<std::string, EventsDB::Priority, std::less<>> priorities;
        if (teleme3dConfig.isMember("priorities")) {
            const auto& src = teleme3dConfig["priorities"];
            if (src.isObject()) {
                const std::map<std::string, EventsDB::Priority> cfgGroups = {
                    {"lowest", EventsDB::LOWEST},
                    {"low", EventsDB::LOW},
                    {"high", EventsDB::HIGH},
                    {"highest", EventsDB::HIGHEST},
                };
                for (const auto& name : src.getMemberNames()) {
                    auto iter = cfgGroups.find(name);
                    if (iter == cfgGroups.end()) {
                        YIO_LOG_WARN("Unsupported group '" << name << "' in priorities config");
                    } else {
                        auto vec = tryGetEmplaceableStringSet<std::set<std::string>>(src, name, {});
                        for (auto name : vec) {
                            priorities[name] = iter->second;
                        }
                    }
                }
            }
        }
        result.cfg.priority = [priorities = std::move(priorities)](const std::string_view& eventName) {
            auto iter = priorities.find(eventName);
            return iter == std::end(priorities) ? EventsDB::DEFAULT : iter->second;
        };
        return result;
    }
} // namespace

Teleme3dEndpoint::Teleme3dEndpoint(
    std::shared_ptr<YandexIO::IDevice> device,
    const std::shared_ptr<ipc::IIpcFactory>& ipcFactory,
    std::unique_ptr<SinksConfigurer> sinksConfigurer)
    : device_(std::move(device))
    , dbQueue_(std::make_shared<quasar::NamedCallbackQueue>("teleme3dEndpoint", 1000))
    , sessionProvider_(makeSession(device_))
    , sinksConfigurer_(std::move(sinksConfigurer))
{
    try {
        const auto dbConfig = getDbConfig(device_);
        mkdir(dbConfig.path.c_str(), 0777);
        eventsDb_ = makeLmdbEventsDb(dbConfig.path, dbConfig.sizeLimit, dbQueue_);
        eventsDb_->setConfig(dbConfig.cfg);
    } catch (const std::runtime_error& e) {
        YIO_LOG_INFO("Failed to open lmdb: " << e.what());
        throw;
    }

    sinksConfigurer_->registerSinks(device_, eventsDb_);

    server_ = ipcFactory->createIpcServer(SERVICE_NAME);
    server_->setMessageHandler([this](auto message, auto& connection) {
        processQuasarMessage(message, connection);
    });
    server_->listenService();
}

ITelemetryEventsDB::Event Teleme3dEndpoint::buildEvent(const TString& name) {
    auto session = sessionProvider_.getAndIncrementSession();
    ITelemetryEventsDB::Event event;
    event.set_name(name);
    event.set_connection_type(connectionType_);
    event.set_type(ITelemetryEventsDB::Event::CLIENT);
    event.set_session_id(session.id);
    event.set_session_start_time(session.startTime);
    event.set_serial_number(session.eventNumber);
    event.set_timestamp(getNowInSeconds());
    if (!userId_.empty()) {
        event.set_account_type("login");
        std::scoped_lock lock(dataMutex_);
        event.set_account_id(TString(userId_));
    }
    return event;
}

void Teleme3dEndpoint::setPassportUid(const std::string& uid) {
    std::scoped_lock lock(dataMutex_);
    userId_ = uid;
}

void Teleme3dEndpoint::processQuasarMessage(const ipc::SharedMessage& message, ipc::IServer::IClientConnection& connection) {
    if (message->has_metrica_message()) {
        YIO_LOG_DEBUG("Got metrica message: " << shortUtf8DebugString(*message));

        const auto& metricaMessage = message->metrica_message();
        if (metricaMessage.has_report_event()) {
            ITelemetryEventsDB::Event event = buildEvent(metricaMessage.report_event());
            if (metricaMessage.has_report_event_json_value()) {
                event.set_value(metricaMessage.report_event_json_value());
            }
            eventsDb_->pushEvent(event);
        }

        if (metricaMessage.has_report_error()) {
            ITelemetryEventsDB::Event event = buildEvent(metricaMessage.report_error());
            event.set_type(ITelemetryEventsDB::Event::ERROR);
            if (metricaMessage.has_report_error_value()) {
                event.set_value(metricaMessage.report_error_value());
            }
            eventsDb_->pushEvent(event);
        }

        if (metricaMessage.has_report_key_value()) {
            const auto& report = metricaMessage.report_key_value();
            ITelemetryEventsDB::Event event = buildEvent(report.event_name());

            auto makePayload = [&report]() {
                Json::Value json;
                for (const auto& kv : report.key_values()) {
                    json[kv.first] = kv.second;
                }
                return jsonToString(json);
            };

            event.set_value(makePayload());
            eventsDb_->pushEvent(event);
        }

        if (metricaMessage.has_app_environment_value()) {
            eventsDb_->updateEnvironmentVar(metricaMessage.app_environment_value().key(),
                                            metricaMessage.app_environment_value().value());
        }

        if (metricaMessage.has_delete_environment_key()) {
            eventsDb_->removeEnvironmentVar(metricaMessage.delete_environment_key());
        }

        if (metricaMessage.has_timezone()) {
            const auto& timezone = metricaMessage.timezone();

            if (timezone.has_timezone_name()) {
                eventsDb_->updateEnvironmentVar("timezone", timezone.timezone_name());
            }
            if (timezone.has_timezone_offset_sec()) {
                sinksConfigurer_->setTimezoneOffsetSec(timezone.timezone_offset_sec());
            }
        }

        if (metricaMessage.has_location()) {
            const auto& location = metricaMessage.location();
            sinksConfigurer_->setLocation(location.latitude(), location.longitude());
        }

        if (metricaMessage.has_network_status()) {
            connectionType_ = metricaMessage.network_status().type();
        }

        if (metricaMessage.has_config()) {
            sinksConfigurer_->updateConfig(parseJson(metricaMessage.config()));
        }

        if (message->has_request_id()) {
            // Send acknowledge
            connection.send(quasar::ipc::buildMessage([&message](auto& msg) {
                msg.set_request_id(message->request_id());
            }));
        }
    }
}
