#include "backend_client_core.h"

#include <yandex_io/libs/base/utils.h>

using namespace quasar;
using HttpResponse = ISimpleHttpClient::HttpResponse;

namespace {
    proto::BackendRequest makeBackendRequest(proto::BackendRequest::Method method, std::string_view tag, const std::string& url, const ISimpleHttpClient::Headers& headers) {
        proto::BackendRequest req;
        req.set_tag(TString(tag));
        req.set_method(method);
        req.set_url(TString(url));
        for (const auto& [name, value] : headers) {
            auto header = req.add_headers();
            header->set_name(TString(name));
            header->set_value(TString(value));
        }
        return req;
    }

    HttpResponse convertResponse(const proto::BackendResponse& response) {
        HttpResponse result;
        result.responseCode = response.response_code();
        result.responseStatus = response.response_status();
        result.contentType = response.content_type();
        result.body = response.body();
        result.contentLength = response.content_length();
        for (const auto& item : response.extra_headers()) {
            result.extraHeaders.emplace_back(item.name(), item.value());
        }
        return result;
    }
} // namespace

BackendClientCore::BackendClientCore(Sender sender)
    : sender_(std::move(sender))
    , timeoutInterval_(std::chrono::seconds(20))
{
}

BackendClientCore::~BackendClientCore() {
    // force timeouted
    std::unordered_map<std::string, std::shared_ptr<TimedPromise>> tmp;
    {
        std::scoped_lock lock(mutex_);
        tmp.swap(pendingRequests_);
    }
    for (auto& [requestId, timedPromise] : tmp) {
        try {
            throw std::runtime_error("Terminating");
        } catch (...) {
            timedPromise->promise.set_exception(std::current_exception());
        }
    }
}

HttpResponse BackendClientCore::head(std::string_view tag, const std::string& /*url*/, const Headers& /*headers*/) {
    throw std::runtime_error("BackendClientCore: head is unsupported yet. tag = " + std::string(tag));
}

HttpResponse BackendClientCore::get(std::string_view tag, const std::string& url, const Headers& headers) {
    auto req = makeBackendRequest(proto::BackendRequest::GET, tag, url, headers);
    auto timedPromise = enqueue(std::move(req));
    return timedPromise->promise.get_future().get();
}

HttpResponse BackendClientCore::post(std::string_view tag, const std::string& url, const std::string& data, const Headers& headers) {
    auto req = makeBackendRequest(proto::BackendRequest::POST, tag, url, headers);
    req.set_data(TString(data));
    auto timedPromise = enqueue(std::move(req));
    return timedPromise->promise.get_future().get();
}

std::shared_ptr<BackendClientCore::TimedPromise> BackendClientCore::enqueue(proto::BackendRequest req) {
    auto requestId = makeUUID();
    auto result = std::make_shared<TimedPromise>();
    result->requested = std::chrono::steady_clock::now();
    sender_(requestId, std::move(req)); // can throw exception
    {
        std::scoped_lock lock(mutex_);
        pendingRequests_.emplace(requestId, result);
    }
    return result;
}

void BackendClientCore::handleResponse(const std::string& requestId, const proto::BackendResponse& response) {
    auto getPromise = [this, &requestId]() -> std::shared_ptr<TimedPromise> {
        std::scoped_lock lock(mutex_);
        auto iter = pendingRequests_.find(requestId);
        if (iter != pendingRequests_.end()) {
            auto result = iter->second;
            pendingRequests_.erase(iter);
            return result;
        }
        return nullptr;
    };

    auto timedPromise = getPromise();
    if (timedPromise) {
        if (response.has_error()) {
            try {
                throw std::runtime_error(response.error());
            } catch (...) {
                timedPromise->promise.set_exception(std::current_exception());
            }
        } else {
            timedPromise->promise.set_value(convertResponse(response));
        }
    }
}

void BackendClientCore::setTimeoutInterval(std::chrono::milliseconds timeout) {
    std::scoped_lock lock(mutex_);
    timeoutInterval_ = timeout;
}

void BackendClientCore::checkTimeouts() {
    std::vector<std::tuple<std::string, std::shared_ptr<TimedPromise>>> timeouted;
    {
        std::scoped_lock lock(mutex_);
        auto iter = std::begin(pendingRequests_);
        auto end = std::end(pendingRequests_);
        const auto now = std::chrono::steady_clock::now();
        while (iter != end) {
            if (now - iter->second->requested > timeoutInterval_) {
                timeouted.emplace_back(iter->first, std::move(iter->second));
                iter = pendingRequests_.erase(iter);
            } else {
                ++iter;
            }
        }
    }
    for (auto& [requestId, timedPromise] : timeouted) {
        try {
            throw std::runtime_error("Timeouted");
        } catch (...) {
            timedPromise->promise.set_exception(std::current_exception());
        }
    }
}
