#include "backend_endpoint.h"

#include <yandex_io/protos/quasar_proto.pb.h>
#include <yandex_io/libs/logging/logging.h>
#include <yandex_io/libs/http_client/http_client.h>

using namespace quasar;

const std::string BackendEndpoint::SERVICE_NAME = "backend";

namespace {

    std::unique_ptr<IHttpClient> makeHttpClientForBackend(std::shared_ptr<YandexIO::IDevice> device, std::shared_ptr<IBackoffRetries> delayTimingsPolicy) {
        std::unique_ptr<HttpClient> httpClient = std::make_unique<HttpClient>("quasar-backend", device);
        httpClient->setTimeout(std::chrono::seconds(20));
        httpClient->setRetriesCount(0);
        httpClient->setDnsCacheTimeout(std::chrono::seconds(0)); // disable dns cache
        httpClient->setReuse(true);
        httpClient->setCalcRetryDelayFunction([delayTimingsPolicy](int retryNum) {
            return delayTimingsPolicy->calcHttpClientRetriesDelay(retryNum);
        });
        return httpClient;
    }

    IHttpClient::Headers convertHeaders(const auto& headers) {
        IHttpClient::Headers result;
        for (const auto& item : headers) {
            result.emplace(item.name(), item.value());
        }
        return result;
    }

    auto buildBackendResponse(const TString& request_id, const IHttpClient::HttpResponse& response, const std::string& error) {
        return ipc::buildMessage([&](auto& message) {
            message.set_request_id(request_id);
            auto answer = message.mutable_backend_response();
            if (error.empty()) {
                answer->set_response_code(response.responseCode);
                answer->set_response_status(TString(response.responseStatus));
                answer->set_content_type(TString(response.contentType));
                answer->set_body(TString(response.body));
                answer->set_content_length(response.contentLength);
                for (auto [name, value] : response.extraHeaders) {
                    auto header = answer->add_extra_headers();
                    header->set_name(TString(name));
                    header->set_value(TString(value));
                }
                answer->set_send_body(response.sendBody);
            } else {
                answer->set_error(TString(error));
            }
        });
    }
} // namespace

BackendEndpoint::BackendEndpoint(std::shared_ptr<YandexIO::IDevice> device,
                                 std::shared_ptr<ipc::IIpcFactory> ipcFactory,
                                 std::shared_ptr<IAuthProvider> authProvider,
                                 std::shared_ptr<IBackoffRetries> delayTimingsPolicy)
    : device_(std::move(device))
    , authProvider_(std::move(authProvider))
    , backendClient_(makeHttpClientForBackend(device_, delayTimingsPolicy), device_->configuration()->getBackendUrl())
    , server_(ipcFactory->createIpcServer(SERVICE_NAME))
{
    server_->setMessageHandler([this](const auto& message, auto& connection) {
        if (message->has_backend_request() && message->has_request_id()) {
            const auto& request = message->backend_request();
            auto answerCb = [weakConnection = connection.weak(), request_id = message->request_id()](const IHttpClient::HttpResponse& response, const std::string& error) {
                if (auto connection = weakConnection.lock()) {
                    connection->send(buildBackendResponse(request_id, response, error));
                    YIO_LOG_DEBUG("Sended answer " << request_id);
                } else {
                    YIO_LOG_DEBUG("Failed to answer, connection died " << request_id);
                }
            };
            startBackendRequest(request, std::move(answerCb));
        }
    });
    server_->listenService();
}

void BackendEndpoint::startBackendRequest(const proto::BackendRequest& request, QueuedBackend::ResponseCallback answerCb) {
    try {
        YIO_LOG_DEBUG("Requesting backend " << request.url());
        auto headers = convertHeaders(request.headers());
        /*
        if (!authToken.empty()) {
            headers.emplace("Authorization", "OAuth " + authToken);
        }
        */
        if (request.has_method() && request.method() == proto::BackendRequest::POST) {
            backendClient_.postCb(request.tag(), backendClient_.getPath(request.url()), std::move(headers), request.data(), answerCb);
        } else {
            backendClient_.getCb(request.tag(), backendClient_.getPath(request.url()), std::move(headers), answerCb);
        }
    } catch (std::exception& err) {
        YIO_LOG_WARN("Failed to request " << request.url() << ' ' << err.what());
        answerCb(IHttpClient::HttpResponse(), std::string("not requested: ") + err.what());
    }
}
