#include "queued_backend.h"

#include <yandex_io/libs/logging/logging.h>
#include <yandex_io/libs/base/named_callback_queue.h>

using namespace quasar;

namespace {
    class QueuedHttpClient: public ISimpleHttpClient {
        QueuedBackend& backend_;

    public:
        QueuedHttpClient(QueuedBackend& backend)
            : backend_(backend)
        {
        }

        static QueuedBackend::ResponseCallback makeCb(std::promise<HttpResponse>& promise) {
            return [&promise](const HttpResponse& response, std::string error) {
                if (error.empty()) {
                    promise.set_value(response);
                } else {
                    try {
                        throw std::runtime_error(error);
                    } catch (...) {
                        promise.set_exception(std::current_exception());
                    }
                }
            };
        }

        HttpResponse get(std::string_view tag, const std::string& url, const Headers& headers) override {
            std::promise<HttpResponse> promise;
            backend_.getCb(std::string(tag), backend_.getPath(url), headers, makeCb(promise));
            return promise.get_future().get();
        }

        HttpResponse post(std::string_view tag, const std::string& url, const std::string& data, const Headers& headers) override {
            std::promise<HttpResponse> promise;
            backend_.postCb(std::string(tag), backend_.getPath(url), headers, data, makeCb(promise));
            return promise.get_future().get();
        }

        HttpResponse head(std::string_view tag, const std::string& /*url*/, const Headers& /*headers*/) override {
            throw std::runtime_error("QueuedBackendClient: Unsupported 'head' method. tag = " + std::string(tag));
        }
    };

} // namespace

bool QueuedBackend::RequestParams::operator<(const RequestParams& params) const {
    if (path == params.path) {
        return post < params.post;
    }
    return path < params.path;
}

QueuedBackend::QueuedBackend(std::shared_ptr<ICallbackQueue> cbQueue, std::unique_ptr<ISimpleHttpClient> httpClient, const std::string& url)
    : backendUrl_(url)
    , httpClient_(std::move(httpClient))
    , cbQueue_(std::move(cbQueue))
{
}

QueuedBackend::QueuedBackend(std::unique_ptr<ISimpleHttpClient> httpClient, const std::string& url)
    : QueuedBackend(std::make_shared<NamedCallbackQueue>("sync-quasar-backend", 100), std::move(httpClient), url)
{
}

std::shared_ptr<ISimpleHttpClient> QueuedBackend::makeHttpClientOverMe() {
    return std::make_shared<QueuedHttpClient>(*this);
}

std::string QueuedBackend::getBackendUrl() const {
    std::scoped_lock lock(mutex_);
    return backendUrl_;
}

std::string QueuedBackend::getPath(const std::string& url) const {
    auto prefix = getBackendUrl();
    if (!url.starts_with(prefix)) {
        throw std::runtime_error("Wrong backend in url: '" + url + "' expect '" + prefix + "'");
    }
    return url.substr(prefix.size());
}

void QueuedBackend::setBackendUrl(const std::string& newBackendUrl) {
    std::scoped_lock lock(mutex_);
    backendUrl_ = newBackendUrl;
}

void QueuedBackend::forceProcessing() {
    processing_ = true;
    cbQueue_->add([this]() {
        try {
            processFirstRequest();
        } catch (const std::exception& e) {
            YIO_LOG_WARN("Unhandled exception during processing request: " << e.what());
        } catch (...) {
            YIO_LOG_WARN("Unhandled unknown exception during processing request");
        }
        if (std::scoped_lock lock(mutex_); !requests_.empty()) {
            cbQueue_->add([this] {
                forceProcessing();
            });
        }
        processing_ = false;
    });
}

void QueuedBackend::processFirstRequest() {
    auto popRequest = [this]() -> std::tuple<RequestParams, std::shared_ptr<Clients>> {
        std::scoped_lock lock(mutex_);
        if (requests_.empty()) {
            return {{}, nullptr};
        }
        auto iter = requestsQueue_.front();
        requestsQueue_.pop_front();
        auto params = iter->first;
        auto clients = iter->second;
        requests_.erase(iter);
        return {params, clients};
    };

    auto [params, clients] = popRequest();
    if (!clients) {
        YIO_LOG_DEBUG("No requests");
        return;
    }

    try {
        auto response = doRequest(params, clients->payload);
        enqueueCallbacks(response, {}, std::move(clients->callbacks));
    } catch (std::runtime_error& err) {
        enqueueCallbacks({}, err.what(), std::move(clients->callbacks));
    } catch (...) {
        enqueueCallbacks({}, "unknown exception", std::move(clients->callbacks));
    }
}

void QueuedBackend::enqueueCallbacks(HttpResponse response, std::string error, std::deque<ResponseCallback> callbacks) {
    if (!callbacks.empty()) {
        cbQueue_->add([response = std::move(response), error = std::move(error), callbacks = std::move(callbacks)]() {
            for (auto& cb : callbacks) {
                try {
                    cb(response, error);
                } catch (...) {
                }
            }
        });
    }
}

QueuedBackend::HttpResponse QueuedBackend::doRequest(const RequestParams& params, const Payload& payload) {
    YIO_LOG_DEBUG("Doing request " << params.path);
    if (params.post) {
        return httpClient_->post(payload.tag, getBackendUrl() + params.path, payload.data, payload.headers);
    }
    return httpClient_->get(payload.tag, getBackendUrl() + params.path, payload.headers);
}

void QueuedBackend::enqueue(RequestParams req, Payload payload, ResponseCallback& cb) {
    std::scoped_lock lock(mutex_);

    auto [iter, inserted] = requests_.emplace(std::move(req), nullptr);
    if (inserted) {
        iter->second = std::make_shared<Clients>();
        requestsQueue_.push_back(iter);
    }
    iter->second->payload = std::move(payload);
    iter->second->callbacks.emplace_back(std::move(cb));
    if (!processing_) {
        forceProcessing();
    }
}

void QueuedBackend::getCb(std::string tag, const std::string& path, Headers headers, ResponseCallback cb) {
    enqueue(RequestParams{.path = path}, Payload{.tag = std::move(tag), .headers = std::move(headers)}, cb);
}

void QueuedBackend::postCb(std::string tag, const std::string& path, Headers headers, std::string data, ResponseCallback cb) {
    enqueue(RequestParams{.path = path, .post = true}, Payload{.tag = std::move(tag), .headers = std::move(headers), .data = std::move(data)}, cb);
}
