#include "bluetooth_backend_config_listener.h"

#include <yandex_io/libs/json_utils/json_utils.h>
#include <yandex_io/libs/logging/logging.h>

#include <util/charset/utf8.h>

YIO_DEFINE_LOG_MODULE("bluetooth");

namespace {

    const std::string CONFIG_NAME = "name";

    constexpr int MAX_NAME_LEN = 64;

} // namespace

using namespace YandexIO;

BluetoothBackendConfigListener::BluetoothBackendConfigListener(std::shared_ptr<Bluetooth> bluetoothImpl)
    : bluetoothImpl_(std::move(bluetoothImpl))
{
}

// NOLINTNEXTLINE(readability-convert-member-functions-to-static)
void BluetoothBackendConfigListener::subscribeOnConfig(const std::shared_ptr<SDKInterface>& sdk) {
    sdk->subscribeToDeviceConfig(CONFIG_NAME);
}

void BluetoothBackendConfigListener::onDeviceConfig(const std::string& configName, const std::string& jsonConfigValue) {
    if (configName != CONFIG_NAME) {
        return;
    }
    auto name = quasar::tryParseJson(jsonConfigValue);
    if (!name.has_value()) {
        YIO_LOG_ERROR_EVENT("BluetoothBackendConfigListener.InvalidBackendConfigJson", "Cant parse device name from " << jsonConfigValue);
        return;
    }
    if (name.value() != Json::nullValue) {
        updateDeviceName(name.value().asString());
    }
}

void BluetoothBackendConfigListener::updateDeviceName(std::string name) {
    if (name.size() >= MAX_NAME_LEN) {
        size_t newSize = 0;
        for (newSize = 0; newSize < name.size();) {
            /* Get size of ut8Character */
            const int utf8Len = UTF8RuneLen(static_cast<unsigned char>(name[newSize]));
            if (utf8Len == 0) {
                YIO_LOG_ERROR_EVENT("BluetoothBackendConfigListener.NonUtf8Name", "Failed to update device name. Not Utf8 symbol. Drop string with new name");
                /* Got 0 char len. Undefined behaviour. Skip it */
                return;
            }
            if (newSize + utf8Len >= MAX_NAME_LEN) {
                /* Reached size limit with valid ut8 characters */
                break;
            }
            newSize += utf8Len;
        }
        /* Truncate string to the last valid character */
        name.resize(newSize);
    }
    if (name.empty()) {
        return;
    }
    bluetoothImpl_->setName(name);
}
