#include "bluetooth_brick_status_listener.h"

using namespace YandexIO;

BluetoothBrickStatusListener::BluetoothBrickStatusListener(std::shared_ptr<BluetoothImplProxy> bluetoothImpl)
    : bluetoothImpl_(std::move(bluetoothImpl))
{
    bluetoothImpl_->enabledChangedSignal().connect(
        [this](bool enabled) {
            notifyBrickStatusChanged(!enabled);
        }, lifetime_);
}

BluetoothBrickStatusListener::~BluetoothBrickStatusListener()
{
    lifetime_.die();
}

void BluetoothBrickStatusListener::notifyBrickStatusChanged(bool status) {
    const auto listeners = getBrickStatusListeners();
    for (const auto& wlistener : listeners) {
        if (auto listener = wlistener.lock()) {
            listener->onBrickStatusChanged(status);
        }
    }
}

void BluetoothBrickStatusListener::addListener(std::weak_ptr<IBrickStatusListener> listener) {
    std::scoped_lock guard(mutex_);
    listeners_.push_back(listener);
}

std::list<std::weak_ptr<IBrickStatusListener>> BluetoothBrickStatusListener::getBrickStatusListeners() {
    std::scoped_lock guard(mutex_);
    return listeners_;
}

void BluetoothBrickStatusListener::onBrickStatusChanged(BrickStatus brickStatus, const std::string& /*brickStatusUrl*/, bool /*subscriptionMode*/) {
    switch (brickStatus) {
        case BrickStatus::BRICK:
        case BrickStatus::BRICK_BY_TTL:
            bluetoothImpl_->disableBluetooth("BRICK_STATUS_LISTENER");
            break;
        case BrickStatus::NOT_BRICK:
            bluetoothImpl_->enableBluetooth("BRICK_STATUS_LISTENER");
            break;
        default:
            break;
    }
}
