#include "bluetooth_media_event_listener.h"

#include <yandex_io/libs/logging/logging.h>

using namespace YandexIO;

BluetoothMediaEventListener::BluetoothMediaEventListener(std::shared_ptr<SDKInterface> sdk, bool blockAlice)
    : sdk_(std::move(sdk))
    , blockAlice_(blockAlice)
    , aliceBlocker_(sdk_, "bluetooth")
{
}

void BluetoothMediaEventListener::addBluetoothListener(std::weak_ptr<BluetoothObserver> listener) {
    std::scoped_lock guard(mutex_);
    btListeners_.push_back(std::move(listener));
}

void BluetoothMediaEventListener::addBluetoothStateListener(std::weak_ptr<IBluetoothStateListener> listener) {
    std::scoped_lock guard(mutex_);
    btStateListeners_.push_back(std::move(listener));
}

std::list<std::weak_ptr<BluetoothObserver>> BluetoothMediaEventListener::getBtListeners() const {
    std::scoped_lock guard(mutex_);
    return btListeners_;
}

std::list<std::weak_ptr<IBluetoothStateListener>> BluetoothMediaEventListener::getBtStateListeners() const {
    std::scoped_lock guard(mutex_);
    return btStateListeners_;
}

void BluetoothMediaEventListener::notifySourceConnected(const std::string& networkAddr) {
    const auto listeners = getBtListeners();
    for (const auto& wlistener : listeners) {
        if (auto listener = wlistener.lock()) {
            listener->onSourceConnected(networkAddr);
        }
    }
}

void BluetoothMediaEventListener::notifySourceDisconnected(const std::string& networkAddr) {
    const auto listeners = getBtListeners();
    for (const auto& wlistener : listeners) {
        if (auto listener = wlistener.lock()) {
            listener->onSourceDisconnected(networkAddr);
        }
    }
}

void BluetoothMediaEventListener::notifyConnectedState(bool state) {
    const auto listeners = getBtStateListeners();
    for (const auto& wlistener : listeners) {
        if (auto listener = wlistener.lock()) {
            listener->onBtConnected(state);
        }
    }
}

void BluetoothMediaEventListener::notifyPlayingState(bool state) {
    const auto listeners = getBtStateListeners();
    for (const auto& wlistener : listeners) {
        if (auto listener = wlistener.lock()) {
            listener->onBtPlaying(state);
        }
    }
}

void BluetoothMediaEventListener::blockAssistant() {
    if (blockAlice_) {
        aliceBlocker_.block();
    }
}

void BluetoothMediaEventListener::unblockAssistant() {
    if (blockAlice_) {
        aliceBlocker_.unblock();
    }
}

void BluetoothMediaEventListener::onBaseEvent(Bluetooth::BaseEvent /*ev*/, const Bluetooth::EventResult& /*res*/) {
}

void BluetoothMediaEventListener::onSourceEvent(Bluetooth::SourceEvent ev, const Bluetooth::EventResult& res) {
    if (ev == Bluetooth::SourceEvent::AVRCP_IN) {
        switch (res.avrcpEvent) {
            case Bluetooth::AVRCP::PLAY_START: {
                sdk_->getPlaybackControlCapability()->play();
                break;
            }
            case Bluetooth::AVRCP::PLAY_PAUSE:
            case Bluetooth::AVRCP::PLAY_STOP: {
                sdk_->getPlaybackControlCapability()->pause();
                break;
            }
            case Bluetooth::AVRCP::PLAY_NEXT: {
                sdk_->getPlaybackControlCapability()->next();
                break;
            }
            case Bluetooth::AVRCP::PLAY_PREV: {
                sdk_->getPlaybackControlCapability()->prev();
                break;
            }
            default:
                break;
        }
    }
}

void BluetoothMediaEventListener::onSinkEvent(Bluetooth::SinkEvent ev, const Bluetooth::EventResult& res) {
    switch (ev) {
        case Bluetooth::SinkEvent::DISCONNECTED: {
            YIO_LOG_INFO("Disconnected event");
            unblockAssistant();
            sdk_->bluetoothMediaSinkPause();
            sdk_->bluetoothSinkDisconnected(res.network.addr, res.network.name);
            notifySourceDisconnected(res.network.addr);
            notifyPlayingState(false);
            if (--connectionsCounter_ == 0) {
                notifyConnectedState(false);
            }
            YIO_LOG_INFO("Connections Counter: " << connectionsCounter_);
            break;
        }
        case Bluetooth::SinkEvent::CONNECTED: {
            YIO_LOG_INFO("Connected event");
            sdk_->bluetoothSinkConnected(res.network.addr, res.network.name);
            notifySourceConnected(res.network.addr);
            if (connectionsCounter_++ == 0) {
                notifyConnectedState(true);
            }
            YIO_LOG_INFO("Connections Counter: " << connectionsCounter_);
            break;
        }
        case Bluetooth::SinkEvent::AVRCP_IN: {
            switch (res.avrcpEvent) {
                case Bluetooth::AVRCP::PLAY_STOP:
                case Bluetooth::AVRCP::PLAY_PAUSE: {
                    YIO_LOG_INFO("Bluetooth stream stopped");
                    unblockAssistant();
                    sdk_->bluetoothMediaSinkPause();
                    notifyPlayingState(false);
                    break;
                } /* PLAY_PAUSE */
                case Bluetooth::AVRCP::PLAY_START: {
                    YIO_LOG_INFO("Bluetooth stream started");
                    blockAssistant();
                    sdk_->bluetoothMediaSinkStart();
                    notifyPlayingState(true);
                    break;
                } /* PLAY_START */
                case Bluetooth::AVRCP::TRACK_META_INFO: {
                    const auto& meta = res.trackInfo;
                    sdk_->bluetoothMediaSinkTrackInfo(meta.title, meta.artist, meta.album, meta.genre, meta.songLenMs, meta.currPosMs);
                } /* TRACK_META_INFO */
                default: {
                    break;
                }
            }
            break;
        }
        default:
            break;
    }
}
