#include "bluetooth_visibility_event_listener.h"

#include <yandex_io/libs/logging/logging.h>

#include <util/system/yassert.h>

using namespace YandexIO;

BluetoothVisibilityEventListener::BluetoothVisibilityEventListener(std::weak_ptr<Bluetooth> bluetoohtImpl, std::optional<std::chrono::milliseconds> fallbackTimeout)
    : bluetoothImpl_(std::move(bluetoohtImpl))
    , fallbackTimeout_(fallbackTimeout)
{
}

BluetoothVisibilityEventListener::~BluetoothVisibilityEventListener() {
    discoveryFallbackManager_.reset();
}

void BluetoothVisibilityEventListener::addListener(std::weak_ptr<BluetoothObserver> listener) {
    listeners_.push_back(std::move(listener));
}

void BluetoothVisibilityEventListener::notifyDiscoveryStart() {
    const auto listeners = getListeners();
    for (const auto& wlistener : listeners) {
        if (auto listener = wlistener.lock()) {
            listener->onDiscoveryStart();
        }
    }
}

void BluetoothVisibilityEventListener::notifyDiscoveryStop() {
    const auto listeners = getListeners();
    for (const auto& wlistener : listeners) {
        if (auto listener = wlistener.lock()) {
            listener->onDiscoveryStop();
        }
    }
}

std::list<std::weak_ptr<BluetoothObserver>> BluetoothVisibilityEventListener::getListeners() const {
    std::scoped_lock guard(mutex_);
    return listeners_;
}

void BluetoothVisibilityEventListener::onBaseEvent(Bluetooth::BaseEvent /*ev*/, const Bluetooth::EventResult& /*res*/) {
}

void BluetoothVisibilityEventListener::onSourceEvent(Bluetooth::SourceEvent /*ev*/, const Bluetooth::EventResult& /*res*/) {
}

void BluetoothVisibilityEventListener::onSinkEvent(Bluetooth::SinkEvent ev, const Bluetooth::EventResult& /*res*/) {
    switch (ev) {
        case Bluetooth::SinkEvent::DISCOVERABLE:
        case Bluetooth::SinkEvent::DISCOVERABLE_CONNECTABLE:
            YIO_LOG_INFO("Discovery Start");
            if (fallbackTimeout_.has_value()) {
                discoveryFallbackManager_ = std::make_unique<quasar::PeriodicExecutor>([wbluetoothImpl = bluetoothImpl_]() {
                    if (auto bluetoothImpl = wbluetoothImpl.lock()) {
                        bluetoothImpl->setVisibility(false, true);
                    }
                }, *fallbackTimeout_, quasar::PeriodicExecutor::PeriodicType::ONE_SHOT);
            }
            notifyDiscoveryStart();
            break;
        case Bluetooth::SinkEvent::NON_VISIBLE:
        case Bluetooth::SinkEvent::CONNECTABLE:
            YIO_LOG_INFO("Discovery Stop");
            discoveryFallbackManager_.reset();
            notifyDiscoveryStop();
            break;
        default:
            break;
    }
}
