#include "bluetooth_impl_proxy.h"

#include <yandex_io/libs/logging/logging.h>

using namespace YandexIO;

class BluetoothImplProxy::Listener: public BluetoothImplProxy::BluetoothEventListener {
public:
    Listener(BluetoothImplProxy* owner)
        : tracker_(owner->lifetime_.tracker())
        , owner_(owner)
    {
    }
    void onBaseEvent(Bluetooth::BaseEvent ev, const Bluetooth::EventResult& /*res*/) override {
        if (auto lock = tracker_.lock()) {
            switch (ev) {
                case Bluetooth::BaseEvent::POWER_ON:
                    owner_->discardIfDisabled("POWER_ON");
                    break;
                default:
                    break;
            }
        }
    }
    void onSourceEvent(Bluetooth::SourceEvent ev, const Bluetooth::EventResult& /*res*/) override {
        if (auto lock = tracker_.lock()) {
            switch (ev) {
                case Bluetooth::SourceEvent::PAIRED:
                    owner_->discardIfDisabled("PAIRED");
                    break;
                case Bluetooth::SourceEvent::AVRCP_IN:
                    owner_->discardIfDisabled("AVRCP_IN");
                    break;
                case Bluetooth::SourceEvent::AVRCP_OUT:
                    owner_->discardIfDisabled("AVRCP_OUT");
                    break;
                default:
                    break;
            }
        }
    }

    void onSinkEvent(Bluetooth::SinkEvent ev, const Bluetooth::EventResult& /*res*/) override {
        if (auto lock = tracker_.lock()) {
            switch (ev) {
                case Bluetooth::SinkEvent::CONNECTED:
                    owner_->discardIfDisabled("CONNECTED");
                    break;
                case Bluetooth::SinkEvent::DISCOVERABLE:
                    owner_->discardIfDisabled("DISCOVERABLE");
                    break;
                case Bluetooth::SinkEvent::CONNECTABLE:
                    owner_->discardIfDisabled("CONNECTABLE");
                    break;
                case Bluetooth::SinkEvent::DISCOVERABLE_CONNECTABLE:
                    owner_->discardIfDisabled("DISCOVERABLE_CONNECTABLE");
                    break;
                case Bluetooth::SinkEvent::AVRCP_IN:
                    owner_->discardIfDisabled("AVRCP_IN");
                    break;
                case Bluetooth::SinkEvent::AVRCP_OUT:
                    owner_->discardIfDisabled("AVRCP_OUT");
                    break;
                default:
                    break;
            }
        }
    }

private:
    quasar::Lifetime::Tracker tracker_;
    BluetoothImplProxy* owner_;
};

BluetoothImplProxy::BluetoothImplProxy(std::shared_ptr<Bluetooth> bluetoothImpl)
    : Bluetooth("proxy")
    , bluetoothImpl_(std::move(bluetoothImpl))
    , listener_(std::make_shared<Listener>(this))
{
    bluetoothImpl_->registerEventListener(listener_);
}

BluetoothImplProxy::~BluetoothImplProxy() {
    listener_.reset();
    lifetime_.die();
}

int BluetoothImplProxy::setName(const std::string& name) {
    if (!isEnabled()) {
        YIO_LOG_INFO("Ignoring setName command. Bluetooth is disabled.")
        return -1;
    }
    return bluetoothImpl_->setName(name);
}

int BluetoothImplProxy::scanNetworks() {
    if (!isEnabled()) {
        YIO_LOG_INFO("Ignoring scanNetworks command. Bluetooth is disabled.")
        return -1;
    }
    return bluetoothImpl_->scanNetworks();
}

int BluetoothImplProxy::stopScanNetworks() {
    if (!isEnabled()) {
        YIO_LOG_INFO("Ignoring stopScanNetworks command. Bluetooth is disabled.")
        return -1;
    }
    return bluetoothImpl_->stopScanNetworks();
}

int BluetoothImplProxy::disconnectAll(BtRole role) {
    if (!isEnabled()) {
        YIO_LOG_INFO("Ignoring disconnectAll command. Bluetooth is disabled.")
        return -1;
    }
    return bluetoothImpl_->disconnectAll(role);
}

int BluetoothImplProxy::pairWithSink(const BtNetwork& network) {
    if (!isEnabled()) {
        YIO_LOG_INFO("Ignoring pairWithSink command. Bluetooth is disabled.")
        return -1;
    }
    return bluetoothImpl_->pairWithSink(network);
}

int BluetoothImplProxy::setVisibility(bool isDiscoverable, bool isConnectable) {
    if (!isEnabled()) {
        YIO_LOG_INFO("Ignoring setVisibility command. Bluetooth is disabled.")
        return -1;
    }
    return bluetoothImpl_->setVisibility(isDiscoverable, isConnectable);
}

int BluetoothImplProxy::asSinkPlayNext(const BtNetwork& network) {
    if (!isEnabled()) {
        YIO_LOG_INFO("Ignoring asSinkPlayNext command. Bluetooth is disabled.")
        return -1;
    }
    return bluetoothImpl_->asSinkPlayNext(network);
}

int BluetoothImplProxy::asSinkPlayPrev(const BtNetwork& network) {
    if (!isEnabled()) {
        YIO_LOG_INFO("Ignoring asSinkPlayPrev command. Bluetooth is disabled.")
        return -1;
    }
    return bluetoothImpl_->asSinkPlayPrev(network);
}

int BluetoothImplProxy::asSinkPlayPause(const BtNetwork& network) {
    if (!isEnabled()) {
        YIO_LOG_INFO("Ignoring asSinkPlayPause command. Bluetooth is disabled.")
        return -1;
    }
    return bluetoothImpl_->asSinkPlayPause(network);
}

int BluetoothImplProxy::asSinkPlayStart(const BtNetwork& network) {
    if (!isEnabled()) {
        YIO_LOG_INFO("Ignoring asSinkPlayStart command. Bluetooth is disabled.")
        return -1;
    }
    return bluetoothImpl_->asSinkPlayStart(network);
}

int BluetoothImplProxy::asSinkSetVolumeAbs(int volume) {
    if (!isEnabled()) {
        YIO_LOG_INFO("Ignoring asSinkSetVolumeAbs command. Bluetooth is disabled.")
        return -1;
    }
    return bluetoothImpl_->asSinkSetVolumeAbs(volume);
}

int BluetoothImplProxy::powerOn() {
    if (!isEnabled()) {
        YIO_LOG_INFO("Ignoring powerOn command. Bluetooth is disabled.")
        return -1;
    }
    return bluetoothImpl_->powerOn();
}

int BluetoothImplProxy::powerOff() {
    if (!isEnabled()) {
        YIO_LOG_INFO("Ignoring powerOff command. Bluetooth is disabled.")
        return -1;
    }
    return bluetoothImpl_->powerOff();
}

void BluetoothImplProxy::freeAudioFocus() {
    if (!isEnabled()) {
        YIO_LOG_INFO("Ignoring freeAudioFocus command. Bluetooth is disabled.")
        return;
    }
    bluetoothImpl_->freeAudioFocus();
}

void BluetoothImplProxy::takeAudioFocus() {
    if (!isEnabled()) {
        YIO_LOG_INFO("Ignoring takeAudioFocus command. Bluetooth is disabled.")
        return;
    }
    bluetoothImpl_->takeAudioFocus();
}

Bluetooth::PowerState BluetoothImplProxy::getPowerState() const {
    return bluetoothImpl_->getPowerState();
}

bool BluetoothImplProxy::isAsSinkPlaying() const {
    return bluetoothImpl_->isAsSinkPlaying();
}

void BluetoothImplProxy::factoryReset() {
    if (!isEnabled()) {
        YIO_LOG_INFO("Ignoring setName command. Bluetooth is disabled.")
        return;
    }
    bluetoothImpl_->factoryReset();
}

void BluetoothImplProxy::registerEventListener(const std::weak_ptr<Bluetooth::BluetoothEventListener>& listener) {
    bluetoothImpl_->registerEventListener(listener);
}

void BluetoothImplProxy::enableBluetooth(std::string_view block) {
    bool emitSignal = false;
    {
        std::lock_guard lock(mutex_);
        bool isBlocked = !blockers_.empty();
        blockers_.erase(std::string{block});
        if (isBlocked && blockers_.empty()) {
            YIO_LOG_INFO("Enable bluetooth");
            bluetoothImpl_->powerOn();
        }
    }
    if (emitSignal) {
        enabledChangedSignal_(true);
    }
}

void BluetoothImplProxy::disableBluetooth(std::string_view block) {
    bool emitSignal = false;
    {
        std::lock_guard lock(mutex_);
        bool isBlocked = !blockers_.empty();
        blockers_.insert(std::string{block});
        if (!isBlocked) {
            YIO_LOG_INFO("Disable bluetooth: " << *blockers_.begin());
            bluetoothImpl_->disconnectAll(Bluetooth::BtRole::ALL);
            bluetoothImpl_->setVisibility(false, false);
            bluetoothImpl_->powerOff();
            emitSignal = true;
        }
    }
    if (emitSignal) {
        enabledChangedSignal_(false);
    }
}

bool BluetoothImplProxy::isEnabled() const {
    std::lock_guard lock(mutex_);
    return blockers_.empty();
}

std::set<std::string> BluetoothImplProxy::blockers() const {
    std::lock_guard lock(mutex_);
    return blockers_;
}

quasar::ISignal<bool>& BluetoothImplProxy::enabledChangedSignal()
{
    return enabledChangedSignal_;
}

void BluetoothImplProxy::discardIfDisabled(std::string_view eventName)
{
    if (!isEnabled()) {
        YIO_LOG_INFO("Reset connection because bluetooth is disabled: " << eventName);
        bluetoothImpl_->disconnectAll(Bluetooth::BtRole::ALL);
        bluetoothImpl_->setVisibility(false, false);
        bluetoothImpl_->powerOff();
    }
}
