#include "bluetooth_media_listener.h"

#include <yandex_io/libs/logging/logging.h>

YIO_DEFINE_LOG_MODULE("bluetooth");

using namespace YandexIO;

BluetoothMediaListener::BluetoothMediaListener(std::shared_ptr<Bluetooth> bluetoothImpl)
    : bluetoothImpl_(std::move(bluetoothImpl))
    , prevTimer_([this]() {
        Bluetooth::BtNetwork stub;
        bluetoothImpl_->asSinkPlayPrev(stub);
    })
{
    brickStatusListener_ = std::make_shared<CallbackBrickStatusListener>([this](bool status) {
        bricked_ = status;
    });
}

std::weak_ptr<IBrickStatusListener> BluetoothMediaListener::getBrickStatusListener() const {
    return brickStatusListener_;
}

void BluetoothMediaListener::onBtPause() {
    if (bricked_) {
        YIO_LOG_INFO("Device is bricked. Bluetooth is unavailable");
        return;
    }
    Bluetooth::BtNetwork stub;
    bluetoothImpl_->asSinkPlayPause(stub);
}

void BluetoothMediaListener::onBtResume() {
    if (bricked_) {
        YIO_LOG_INFO("Device is bricked. Bluetooth is unavailable");
        return;
    }
    Bluetooth::BtNetwork stub;
    bluetoothImpl_->asSinkPlayStart(stub);
}

void BluetoothMediaListener::onBtNext() {
    if (bricked_) {
        YIO_LOG_INFO("Device is bricked. Bluetooth is unavailable");
        return;
    }
    Bluetooth::BtNetwork stub;
    bluetoothImpl_->asSinkPlayNext(stub);
}

void BluetoothMediaListener::onBtPrev(bool forced) {
    if (bricked_) {
        YIO_LOG_INFO("Device is bricked. Bluetooth is unavailable");
        return;
    }
    Bluetooth::BtNetwork stub;
    bluetoothImpl_->asSinkPlayPrev(stub);
    if (forced) {
        /* HACK: If we send these events too quickly, some Android devices still interpret them as single click */
        try {
            prevTimer_.start(std::chrono::milliseconds(100));
        } catch (const std::exception& e) {
            YIO_LOG_ERROR_EVENT("BluetoothMediaListener.Prev", "Failed to start timer" << e.what());
        }
    }
}

void BluetoothMediaListener::onBtTakeAudioFocus() {
    if (bricked_) {
        YIO_LOG_INFO("Device is bricked. Bluetooth is unavailable");
        return;
    }
    bluetoothImpl_->takeAudioFocus();
}

void BluetoothMediaListener::onBtFreeAudioFocus() {
    if (bricked_) {
        YIO_LOG_INFO("Device is bricked. Bluetooth is unavailable");
        return;
    }
    bluetoothImpl_->freeAudioFocus();
}

void BluetoothMediaListener::onBtDisconnectAll() {
    if (bricked_) {
        YIO_LOG_INFO("Device is bricked. Bluetooth is unavailable");
        return;
    }
    bluetoothImpl_->disconnectAll(Bluetooth::BtRole::ALL);
}
