#include "bluetooth_module.h"

#include <yandex_io/libs/logging/logging.h>
#include <yandex_io/protos/quasar_proto.pb.h>

using namespace YandexIO;

BluetoothModule::BluetoothModule(std::shared_ptr<Bluetooth> bluetoothImpl,
                                 std::shared_ptr<quasar::IStereoPairProvider> stereoPairProvider,
                                 const std::shared_ptr<SDKInterface>& sdk,
                                 const Settings& settings)
    : bluetoothImplProxy_(std::make_shared<BluetoothImplProxy>(std::move(bluetoothImpl)))
    , bluetoothController_(std::make_shared<BluetoothController>(bluetoothImplProxy_))
    , stereoPairProvider_(std::move(stereoPairProvider))
{
    if (settings.useDirectiveObserver) {
        bluetoothDirectiveObserver_ = std::make_shared<BluetoothDirectiveObserver>(bluetoothImplProxy_, settings.powerOffByDirective);
        sdk->addDirectiveObserver(bluetoothDirectiveObserver_);
    }

    if (settings.useBluetoothMediaListener) {
        bluetoothMediaListener_ = std::make_shared<BluetoothMediaListener>(bluetoothImplProxy_);
        sdk->addBluetoothMediaObserver(bluetoothMediaListener_);
    }

    bluetoothMediaEventListener_ = std::make_shared<BluetoothMediaEventListener>(sdk, settings.blockAlice);
    bluetoothVisibilityEventListener_ = std::make_shared<BluetoothVisibilityEventListener>(bluetoothImplProxy_, settings.visibilityFallbackTimeout);
    bluetoothFactoryResetEventListener_ = std::make_shared<BluetoothFactoryResetEventListener>();
    bluetoothVolumeEventListerner_ = std::make_shared<BluetoothVolumeEventListerner>();
    bluetoothImplProxy_->registerEventListener(bluetoothVolumeEventListerner_);
    bluetoothImplProxy_->registerEventListener(bluetoothVisibilityEventListener_);
    bluetoothImplProxy_->registerEventListener(bluetoothMediaEventListener_);
    bluetoothImplProxy_->registerEventListener(bluetoothFactoryResetEventListener_);

    if (settings.bluetoothOnStartSetup) {
        bluetoothModeOnStartSetup_ = std::make_shared<BluetoothModeOnStartSetup>(bluetoothImplProxy_);
        sdk->addDeviceModeObserver(bluetoothModeOnStartSetup_);
        bluetoothSDKStateObserver_ = std::make_shared<BluetoothSDKStateObserver>(sdk);
        sdk->addSDKStateObserver(bluetoothSDKStateObserver_);
        bluetoothMediaEventListener_->addBluetoothListener(bluetoothSDKStateObserver_);
    }

    if (settings.useBackendConfigObserver) {
        bluetoothBackendConfigListener_ = std::make_shared<BluetoothBackendConfigListener>(bluetoothImplProxy_);
        sdk->addBackendConfigObserver(bluetoothBackendConfigListener_);
        bluetoothBackendConfigListener_->subscribeOnConfig(sdk);
    }

    if (settings.useBrickStateListener) {
        bluetoothBrickStatusListener_ = std::make_shared<BluetoothBrickStatusListener>(bluetoothImplProxy_);
        if (settings.useBluetoothMediaListener) {
            bluetoothBrickStatusListener_->addListener(bluetoothMediaListener_->getBrickStatusListener());
        }
        sdk->addBrickStatusObserver(bluetoothBrickStatusListener_);
    }

    bluetoothVolumeManagerListener_ = std::make_shared<BluetoothVolumeManagerListener>(bluetoothImplProxy_, settings.maxVolume);

    if (settings.useSteamOutManager) {
        bluetoothStreamOutManager_ = std::make_shared<BluetoothStreamOutManager>(settings.streamOutManagerSettings.device,
                                                                                 settings.streamOutManagerSettings.authProvider,
                                                                                 bluetoothImplProxy_,
                                                                                 settings.streamOutManagerSettings.backendUrl,
                                                                                 settings.streamOutManagerSettings.deviceID,
                                                                                 settings.streamOutManagerSettings.platform);
        sdk->addBackendConfigObserver(bluetoothStreamOutManager_);
        sdk->addPushNotificationObserver(bluetoothStreamOutManager_);
        bluetoothStreamOutManager_->subscribeOnConfig(sdk);
        bluetoothImplProxy_->registerEventListener(bluetoothStreamOutManager_);

        bluetoothStreamOutDirectiveObserver_ = std::make_shared<BluetoothStreamOutDirectiveObserver>(bluetoothStreamOutManager_, bluetoothImplProxy_);
        sdk->addDirectiveObserver(bluetoothStreamOutDirectiveObserver_);
    }

    if (settings.telemetry) {
        bluetoothTelemetry_ = std::make_shared<BluetoothTelemetry>(settings.telemetry, bluetoothImplProxy_->getPowerState() == Bluetooth::PowerState::ON, settings.telemetrySettings);
        bluetoothImplProxy_->registerEventListener(bluetoothTelemetry_);
    }

    if (stereoPairProvider_) {
        bool isFollower = stereoPairProvider_->stereoPairState().value()->isFollower();
        stereoPairProvider_->stereoPairState().connect(
            [this, isFollower{isFollower}](const auto& state) mutable {
                if (isFollower != state->isFollower()) {
                    isFollower = state->isFollower();
                    if (isFollower) {
                        bluetoothImplProxy_->disableBluetooth("STEREO_PAIR");
                    } else {
                        bluetoothImplProxy_->enableBluetooth("STEREO_PAIR");
                    }
                }
            }, lifetime_);
    }
}

BluetoothModule::~BluetoothModule()
{
    lifetime_.die();
}

void BluetoothModule::addBluetoothStateListener(std::weak_ptr<IBluetoothStateListener> listener) {
    bluetoothMediaEventListener_->addBluetoothStateListener(listener);
}

void BluetoothModule::addBluetoothListener(std::weak_ptr<BluetoothObserver> listener) {
    bluetoothMediaEventListener_->addBluetoothListener(listener);
    bluetoothVolumeEventListerner_->addBluetoothListener(listener);
    bluetoothFactoryResetEventListener_->addListener(listener);
    bluetoothVisibilityEventListener_->addListener(listener);
}

std::shared_ptr<BluetoothVolumeManagerListener> BluetoothModule::getVolumeManagerListener() const {
    return bluetoothVolumeManagerListener_;
}

std::shared_ptr<BluetoothController> BluetoothModule::getBluetoothController() const {
    return bluetoothController_;
}
