#include "bluetooth_stream_out_manager.h"

#include <yandex_io/libs/base/utils.h>
#include <yandex_io/libs/device/device.h>
#include <yandex_io/libs/ipc/i_connector.h>
#include <yandex_io/libs/json_utils/json_utils.h>
#include <yandex_io/libs/logging/logging.h>
#include <yandex_io/libs/telemetry/telemetry.h>
#include <yandex_io/protos/quasar_proto.pb.h>

#include <chrono>
#include <cstring>
#include <memory>
#include <mutex>

YIO_DEFINE_LOG_MODULE("bluetooth");

using namespace quasar;
using namespace quasar::proto;
using namespace YandexIO;

namespace {
    bool ignoreCaseStringsCompare(std::string_view str1, std::string_view str2) {
        return std::equal(str1.begin(), str1.end(), str2.begin(), str2.end(), [](unsigned char a, unsigned char b)
                          {
                              return std::tolower(a) == std::tolower(b);
                          });
    }
} // namespace

BluetoothStreamOutManager::BluetoothStreamOutManager(
    std::shared_ptr<YandexIO::IDevice> device,
    std::shared_ptr<quasar::IAuthProvider> authProvider,
    std::shared_ptr<Bluetooth> bluetoothImpl,
    std::string backendUrl,
    std::string deviceId,
    std::string platform)
    : device_(std::move(device))
    , authProvider_(std::move(authProvider))
    , bluetoothImpl_(std::move(bluetoothImpl))
    , httpClient_{"bluetooth-quasar-backend", device_}
    , backendUrl_(std::move(backendUrl))
    , deviceId_(std::move(deviceId))
    , platform_(std::move(platform))
    , isStreamOutEnabled_(false)
    , currPairedIdx_(-1)
    , pairRetries_(0)
    , state_(BluetoothDeviceState::IDLE)
    , threadExists_(true)
{
    /* Backend url should not be empty */
    Y_VERIFY(!backendUrl_.empty());

    httpClient_.setTimeout(std::chrono::milliseconds{10000});
    httpClient_.setRetriesCount(0);

    tasksThread_ = std::thread(&BluetoothStreamOutManager::tasksThread, this);
}

BluetoothStreamOutManager::~BluetoothStreamOutManager()
{
    threadExists_ = false;
    std::unique_lock<std::mutex> lock(idleStateMutex_);
    state_ = BluetoothDeviceState::IDLE;
    lock.unlock();
    idleCondVar_.notify_one(); /* wake up waitIdleEvent */

    /* Wake up takeTask() */
    BluetoothTask task(BluetoothTask::Task::BLUETOOTH_OFF);
    taskManager_.addTask(task);

    /* add task so thread will wake up */
    tasksThread_.join();
}

void BluetoothStreamOutManager::onScanProgress(const Bluetooth::EventResult& res)
{
    std::unique_lock<std::mutex> lock(mutex_);
    auto stackCopy = scanIdsStack_;
    lock.unlock();
    if (!stackCopy.empty()) {
        sendScanResult(res, stackCopy);
    }
}

void BluetoothStreamOutManager::onScanDone(const Bluetooth::EventResult& res)
{
    updateBluetoothState(BluetoothDeviceEvent::STOP_SCANNING);
    /* Scan is done. Send/Drop all scan ids */
    std::lock_guard<std::mutex> lock(mutex_);
    sendScanResult(res, scanIdsStack_, true /* DONE */);
}

void BluetoothStreamOutManager::sendScanResult(const Bluetooth::EventResult& res, std::stack<std::string>& scanIdsStack, bool isDone)
{
    const HttpClient::Headers headers = {{"Content-Type", "application/json"},
                                         {"Authorization", "OAuth " + authProvider_->ownerAuthInfo().value()->authToken}};

    std::stringstream ss;
    /* Use second version of scan_result handler to send scan_status */
    ss << backendUrl_ << "/v2/scan_results?device_id=" << deviceId_ << "&platform=" << platform_ << "&scan_id=";
    const std::string url = ss.str();
    Json::Value scanResults;
    if (res.result == Bluetooth::EventResult::Result::OK) {
        const Json::Value networksJsonArray = createNetworksJsonArray(res.scanResult);
        scanResults["networks"] = networksJsonArray;
        /* Set up scan status, so webview will know about device scan progress */
        scanResults["scan_status"] = isDone ? "DONE" : "PROGRESS";
    } else {
        YIO_LOG_ERROR_EVENT("BluetoothStreamOutManager.FailedScanNetworks", "Bluetooth scan networks error...");
        /* Set up empty valid json (backend requirement) */
        scanResults["networks"] = "{}";
        /* Bluetooth Scan failed. Notify WebView about it */
        scanResults["scan_status"] = "FAILED";
    }
    const std::string scanResultStr = jsonToString(scanResults);
    YIO_LOG_INFO("Send Bluetooth Scan Result: " << scanResultStr);
    /* send scanned networks */
    while (!scanIdsStack.empty()) {
        const std::string scanId = scanIdsStack.top();
        scanIdsStack.pop();

        const std::string finalUrl = url + scanId;
        try {
            const HttpClient::HttpResponse response = httpClient_.post("scan-results", finalUrl, scanResultStr, headers);
            if (!isSuccessHttpCode(response.responseCode)) {
                const auto body = tryParseJson(response.body);
                if (body.has_value() && (*body)["message"] == "AUTH_TOKEN_INVALID") {
                    authProvider_->requestAuthTokenUpdate("BluetoothStreamOutManager scan-results");
                }
                YIO_LOG_ERROR_EVENT("BluetoothStreamOutManager.FailedSendScanResults", "Send bluetooth networks Http error. Code: " << response.responseCode);
            }
        } catch (const std::exception& e) {
            YIO_LOG_ERROR_EVENT("BluetoothStreamOutManager.FailedSendScanResults", "Can't reach backend: " << finalUrl);
        }
    }
}

Json::Value BluetoothStreamOutManager::createNetworksJsonArray(const std::set<Bluetooth::BtNetwork>& networks) {
    if (networks.empty()) {
        return Json::arrayValue;
    }
    Json::Value networksJson;
    for (const auto& i : networks) {
        if (i.role == Bluetooth::BtRole::SINK) {
            Json::Value network;
            network["mac"] = i.addr;
            network["name"] = i.name;
            networksJson.append(network);
        }
    }
    return networksJson;
}

// NOLINTNEXTLINE(readability-convert-member-functions-to-static)
void BluetoothStreamOutManager::subscribeOnConfig(const std::shared_ptr<SDKInterface>& sdk) {
    sdk->subscribeToDeviceConfig("bluetooth_state");
    sdk->subscribeToDeviceConfig("bluetooth_paired_devices");
}

void BluetoothStreamOutManager::onDeviceConfig(const std::string& configName, const std::string& jsonConfigValue) {
    bool oldIsStreamOutEnabled = isStreamOutEnabled_;
    if (configName == "bluetooth_state") {
        const auto connectToAcoustics = quasar::tryParseJson(jsonConfigValue);
        if (connectToAcoustics.value_or(Json::nullValue) == Json::nullValue) {
            return;
        }
        if (ignoreCaseStringsCompare(connectToAcoustics.value().asString(), "on") && !isStreamOutEnabled_) {
            YIO_LOG_INFO("User want connect to acoustics");
            isStreamOutEnabled_ = true;
        } else if (ignoreCaseStringsCompare(connectToAcoustics.value().asString(), "off") && isStreamOutEnabled_) {
            YIO_LOG_INFO("User do not want connect to acoustics");
            isStreamOutEnabled_ = false;

            BluetoothTask task;
            task.task = BluetoothTask::Task::DISCONNECT;
            taskManager_.addTask(task);

            /* For StreamOut only devices there is no need to keep Bluetooth ON until user want to connect
             * to external acoustics.
             */
            if (device_->hal()->getBluetoothCapabilities().isStreamOutOnly()) {
                task.task = BluetoothTask::Task::BLUETOOTH_OFF;
                taskManager_.addTask(task);
            }
            return;
        }
    } else if (configName == "bluetooth_paired_devices") {
        const auto bluetoothPairedDevices = quasar::tryParseJson(jsonConfigValue);
        if (bluetoothPairedDevices.value_or(Json::nullValue) == Json::nullValue) {
            YIO_LOG_INFO("Failed to parse Json");
            return;
        }
        if (oldIsStreamOutEnabled == isStreamOutEnabled_ &&
            bluetoothPairedDevicesCache_ == jsonToString(bluetoothPairedDevices.value())) {
            /* Config changed, but bluetooth_paired_devices and bluetooth_state didn't. So skip it.
             * It's probably user changed name
             */
            YIO_LOG_INFO("Bluetooth Paired Devices config didn't change. Skip it");
            return;
        }
        bluetoothPairedDevicesCache_ = jsonToString(bluetoothPairedDevices.value());

        std::lock_guard<std::mutex> guard(mutex_);

        pairedNetworks_.clear();
        if (!bluetoothPairedDevices.value().empty()) {
            device_->telemetry()->putAppEnvironmentValue("user_set_external_acoustics", "1");
            for (Json::Value::ArrayIndex i = 0; i < bluetoothPairedDevices.value().size(); ++i) {
                const Json::Value& btNetwork = bluetoothPairedDevices.value().get(i, Json::Value());
                if (btNetwork.isMember("mac")) {
                    const std::string& mac = btNetwork["mac"].asString();
                    const std::string& name = btNetwork["name"].asString();
                    pairedNetworks_.emplace_back(name, mac, Bluetooth::BtRole::SINK);
                }
            }
            /* Connect to network if currently paired mac is different */
            if (isStreamOutEnabled_ && currentlyPairedMac_ != pairedNetworks_[0].addr) {
                BluetoothTask task;
                if (bluetoothImpl_->getPowerState() == Bluetooth::PowerState::OFF) {
                    /* Turn on bluetooth before pairing */
                    task.task = BluetoothTask::Task::BLUETOOTH_ON;
                    taskManager_.addTask(task);
                }
                currPairedIdx_ = 0;
                pairRetries_ = 0;
                task.task = BluetoothTask::Task::PAIR_WITH;
                task.network_ = pairedNetworks_[0];

                taskManager_.addTask(task);
            }
        } else {
            YIO_LOG_INFO("User removed all devices to pair: disconnect if currently connected");
            device_->telemetry()->putAppEnvironmentValue("user_set_external_acoustics", "0");
            /* we assume that disconnect should work */
            BluetoothTask task;
            task.task = BluetoothTask::Task::DISCONNECT;
            taskManager_.addTask(task);
        }
    }
}

void BluetoothStreamOutManager::onPushNotification(const std::string& /*operation*/, const std::string& messageStr) {
    YIO_LOG_INFO("Recived push notification");
    if (!isStreamOutEnabled_) {
        YIO_LOG_INFO("Stream out is disabled");
        return;
    }
    Json::Value messageJson = tryParseJson(messageStr).value_or(Json::nullValue);
    if (messageJson == Json::nullValue) {
        YIO_LOG_INFO("Failed to parse");
        return;
    }
    if (messageJson.isMember("remote_procedure_call")) {
        const Json::Value& procedure = messageJson["remote_procedure_call"];
        if (procedure.isMember("name") && procedure["name"].asString() == "start_bluetooth_scan") {
            if (procedure.isMember("args")) {
                const Json::Value& args = procedure["args"];
                /* Currently startBluetoothProcedure has only one arg, so get it */
                const Json::Value& arg0 = args.get(Json::Value::ArrayIndex(0), Json::Value());
                /* Validate Args */
                if (!arg0.empty() && arg0.isMember("name") &&
                    (arg0["name"].asString() == "scan_id") && arg0.isMember("value") && !arg0["value"].isNull()) {
                    {
                        std::lock_guard<std::mutex> guard(mutex_);
                        scanIdsStack_.push(arg0["value"].asString());
                    }
                    BluetoothTask task;
                    if (bluetoothImpl_->getPowerState() == Bluetooth::PowerState::OFF) {
                        task.task = BluetoothTask::Task::BLUETOOTH_ON;
                        taskManager_.addTask(task);
                    }

                    task.task = BluetoothTask::Task::SCAN_NETWORKS;
                    taskManager_.addTask(task);
                    /* User called for networks scan. Report event */
                    YIO_LOG_INFO("Start bluetooth scan");
                    device_->telemetry()->reportEvent("start_bluetooth_scan");
                }
            }
        } /* start_scan call */
    }     /* procedure_call */
}

void BluetoothStreamOutManager::onSourceConnectionEvent(Bluetooth::SourceEvent ev, Bluetooth::EventResult res)
{
    std::unique_lock<std::mutex> lock(mutex_);

    if (res.result == Bluetooth::EventResult::Result::OK) {
        if (ev == Bluetooth::SourceEvent::PAIRED) {
            YIO_LOG_INFO("Bluetooth paired with: " << res.network.name << ' ' << res.network.addr);
            device_->telemetry()->putAppEnvironmentValue("is_bt_connected_to_sink", "1");
            device_->telemetry()->reportEvent("bt_connected_to_sink");
            currentlyPairedMac_ = res.network.addr;
            pairRetries_ = 0;
            updateBluetoothState(BluetoothDeviceEvent::STOP_CONNECTING);
            if (pairedNetworks_.empty() || !isStreamOutEnabled_) {
                /* User deleted all networks faster than we got connection event or disabled bluetooth*/
                BluetoothTask task;
                task.task = BluetoothTask::Task::DISCONNECT;
                taskManager_.addTask(task);
            }
            /* successfully paired */
        } else if (ev == Bluetooth::SourceEvent::DISCONNECTED) {
            /* BT successfully disconnected */
            device_->telemetry()->putAppEnvironmentValue("is_bt_connected_to_sink", "0");
            device_->telemetry()->reportEvent("bt_disconnected_from_sink");
            YIO_LOG_INFO("Bluetooth disconnected");
            currentlyPairedMac_.clear();
            updateBluetoothState(BluetoothDeviceEvent::STOP_DISCONNECTING);
            if (!pairedNetworks_.empty()) {
                performReconnect();
            }
        }
    } else {
        /* Connection error */
        if (ev == Bluetooth::SourceEvent::PAIRED) {
            updateBluetoothState(BluetoothDeviceEvent::STOP_CONNECTING);
            YIO_LOG_ERROR_EVENT("BluetoothStreamOutManager.FailedPairing", "Bluetooth pair failed");
            if (!pairedNetworks_.empty()) {
                performReconnect();
            }
        } else if (ev == Bluetooth::SourceEvent::DISCONNECTED) {
            YIO_LOG_INFO("Bluetooth disconnect failed");
            pairRetries_ = 0;
            updateBluetoothState(BluetoothDeviceEvent::STOP_DISCONNECTING);

            BluetoothTask task;
            task.task = BluetoothTask::Task::DISCONNECT;
            taskManager_.addTask(task);
        }
    }
}

/* Should be called under mutex_ */
void BluetoothStreamOutManager::performReconnect()
{
    if (shouldConnect()) {
        reconnectExecutor_.reset(nullptr);
        /* Add reconnect task, so will reconnect will be scheduled in 20 seconds */
        auto reconnectLambda = [this] {
            if (!shouldConnect()) {
                return;
            }
            YIO_LOG_WARN("Call for bluetooth reconnect: " << (int)bluetoothImpl_->getPowerState());
            std::lock_guard<std::mutex> guard(mutex_);

            ++pairRetries_;
            if (pairRetries_ >= PAIR_RETRIES_MAX_COUNT) {
                pairRetries_ = 0;
                /* Switch to the next network to pair */
                ++currPairedIdx_;
                if (currPairedIdx_ >= (int)pairedNetworks_.size()) {
                    currPairedIdx_ = 0;
                }
            }

            if (!pairedNetworks_.empty()) {
                BluetoothTask task;
                task.task = BluetoothTask::Task::RECONNECT;

                task.network_ = pairedNetworks_[currPairedIdx_];
                taskManager_.addTask(task);
            }
        };
        /* Create one shot Periodic executor that will call for reconnect after 20 sec delay.
         * So this thread will be not blocked
         */
        reconnectExecutor_ = std::make_unique<PeriodicExecutor>(reconnectLambda, std::chrono::seconds(20), PeriodicExecutor::PeriodicType::ONE_SHOT);
    }
}

void BluetoothStreamOutManager::onBaseEvent(Bluetooth::BaseEvent ev, const Bluetooth::EventResult& res)
{
    switch (ev) {
        case Bluetooth::BaseEvent::SCANNING:
            onScanProgress(res);
            break;
        case Bluetooth::BaseEvent::SCANNED: {
            onScanDone(res);
            break;
        }
        default: {
            break;
        }
    }
}

void BluetoothStreamOutManager::onSourceEvent(Bluetooth::SourceEvent ev, const Bluetooth::EventResult& res)
{
    switch (ev) {
        case Bluetooth::SourceEvent::DISCONNECTED:
        case Bluetooth::SourceEvent::PAIRED: {
            onSourceConnectionEvent(ev, res);
            break;
        }
        default: {
            break;
        }
    }
}

void BluetoothStreamOutManager::tasksThread()
{
    while (threadExists_) {
        /* Do not get any task until bluetooth state is not idle */
        waitIdle();
        if (!threadExists_) {
            /* BluetoothStreamOutManager is dying */
            break;
        }
        /* Wait for task to take */
        taskManager_.waitTask();

        auto task = taskManager_.takeTask();
        if (!threadExists_) {
            /* BluetoothStreamOutManager is dying */
            break;
        }

        YIO_LOG_INFO("Bluetooth Got Task: " << (int)task.task);
        switch (task.task) {
            case BluetoothTask::Task::BLUETOOTH_ON: {
                YIO_LOG_INFO("Turn Bluetooth ON");
                bluetoothImpl_->powerOn();
                break;
            }
            case BluetoothTask::Task::BLUETOOTH_OFF: {
                YIO_LOG_INFO("Turn Bluetooth OFF");
                bluetoothImpl_->powerOff();
                break;
            }
            case BluetoothTask::Task::RECONNECT:
            case BluetoothTask::Task::PAIR_WITH: {
                std::lock_guard<std::mutex> guard(mutex_);
                /* Already connected to this bt device */
                if (currentlyPairedMac_ != task.network_.addr) {
                    updateBluetoothState(BluetoothDeviceEvent::START_CONNECTING);
                    bluetoothImpl_->pairWithSink(task.network_);
                } else {
                    YIO_LOG_WARN("Already connected to this device");
                }
                break;
            }
            case BluetoothTask::Task::SCAN_NETWORKS: {
                updateBluetoothState(BluetoothDeviceEvent::START_SCANNING);
                bluetoothImpl_->scanNetworks();
                break;
            }
            case BluetoothTask::Task::DISCONNECT: {
                std::lock_guard<std::mutex> guard(mutex_);
                if (!currentlyPairedMac_.empty()) {
                    updateBluetoothState(BluetoothDeviceEvent::START_DISCONNECTING);
                    /* Disconnect all currently connected sink devices */
                    bluetoothImpl_->disconnectAll(Bluetooth::BtRole::SINK);
                } else { /* empty current connection. No need to disconnect */
                    YIO_LOG_INFO("Empty Currently Paired Mac. Skip It");
                }
                break;
            }
            default: {
            }
        }
    }
}

/* Wait until current state will IDLE
 * So wait until PAIR of DISCONNECT commands ends
 */
void BluetoothStreamOutManager::waitIdle()
{
    std::unique_lock<std::mutex> lock(idleStateMutex_);
    if (state_ == BluetoothDeviceState::IDLE) {
        return;
    } else {
        /**
         * Number is kind of empirical. i.e.: DEXP need about 6 sec to connect to SINK
         */
        static constexpr int WAIT_IDLE_SEC = 10;
        auto res = idleCondVar_.wait_for(lock, std::chrono::seconds(WAIT_IDLE_SEC), [this]() {
            return state_ == BluetoothDeviceState::IDLE;
        });
        if (!res) {
            /* The state Manager waited didn't came. So don't block thread and allow user to retry */
            YIO_LOG_ERROR_EVENT("BluetoothStreamOutManager.TimeoutWaitingForIdleState", "BluetoothStreamOut state is not IDLE! Drop current state!");
            state_ = BluetoothDeviceState::IDLE;
        }
    }
}

/* This state machine should prevent extra Pairing and Disconnecting commands
 * It can be extended for SCANNING also
 * It should be guaranteed (by bt API) that STOP_CONNECTING and STOP_DISCONNECTED will be used,
 * otherwise IDLE state will not be reached
 */
void BluetoothStreamOutManager::updateBluetoothState(BluetoothDeviceEvent ev)
{
    std::unique_lock<std::mutex> lock(idleStateMutex_);
    YIO_LOG_INFO("Update State. Current State: " << (int)state_ << ". Event: " << (int)ev);
    switch (ev) {
        case BluetoothDeviceEvent::START_CONNECTING: {
            if (state_ == BluetoothDeviceState::IDLE) {
                state_ = BluetoothDeviceState::CONNECTING;
            } else {
                YIO_LOG_ERROR_EVENT("BluetoothStreamOutManager.InvalidNonIdleStateForPairing", "Bluetooth Try to start pair in NOT IDLE state: " << (int)state_);
            }
            break;
        }
        case BluetoothDeviceEvent::STOP_CONNECTING: {
            if (state_ == BluetoothDeviceState::CONNECTING) {
                state_ = BluetoothDeviceState::IDLE;
                lock.unlock();
                idleCondVar_.notify_one();
            } /* Ignore other event's. i.e.: disconnect
               * Wait until paired
               */
            break;
        }
        case BluetoothDeviceEvent::START_DISCONNECTING: {
            if (state_ == BluetoothDeviceState::IDLE) {
                state_ = BluetoothDeviceState::DISCONNECTING;
            } else {
                YIO_LOG_ERROR_EVENT("BluetoothStreamOutManager.InvalidNonIdleStateForDisconnect", "Bluetooth Try to disconnect in NOT IDLE state: " << (int)state_);
            }
            break;
        }
        /* STOP_DISCONNECTING can happen without START_DISCONNECTING
         * i.e.: Sink device disconnected (turned off)
         */
        case BluetoothDeviceEvent::STOP_DISCONNECTING: {
            if (state_ == BluetoothDeviceState::DISCONNECTING) {
                state_ = BluetoothDeviceState::IDLE;
                lock.unlock();
                idleCondVar_.notify_one();
            }
            break;
        }
        /* Assume that scanning does not affects others functions */
        case BluetoothDeviceEvent::START_SCANNING: {
            if (state_ != BluetoothDeviceState::IDLE) {
                YIO_LOG_ERROR_EVENT("BluetoothStreamOutManager.InvalidNonIdleStateForStartScanning", "Bluetooth Try to start scanning in NOT IDLE state: " << (int)state_);
            }
            break;
        }
        case BluetoothDeviceEvent::STOP_SCANNING: {
            break;
        }
    }
    YIO_LOG_DEBUG("New state: " << (int)state_);
}

void BluetoothStreamOutManager::runStreamOut()
{
    std::lock_guard<std::mutex> guard(mutex_);
    if (isStreamOutEnabled_ && !pairedNetworks_.empty() && currentlyPairedMac_ != pairedNetworks_[0].addr) {
        YIO_LOG_INFO("Run Stream Out by VOICE (connect to: " << pairedNetworks_[0].addr << ')');
        currPairedIdx_ = 0;
        pairRetries_ = 0;
        BluetoothTask task;
        task.task = BluetoothTask::Task::PAIR_WITH;
        task.network_ = pairedNetworks_[0];
        taskManager_.addTask(task);
    }
}

bool BluetoothStreamOutManager::shouldConnect() const {
    return isStreamOutEnabled_ && bluetoothImpl_->getPowerState() == Bluetooth::PowerState::ON;
}

void BluetoothStreamOutManager::stopStreamOut()
{
}
