#pragma once

#include "bluetooth_tasks_manager.h"

#include <yandex_io/interfaces/auth/i_auth_provider.h>
#include <yandex_io/libs/base/utils.h>
#include <yandex_io/libs/bluetooth/bluetooth.h>
#include <yandex_io/libs/device/device.h>
#include <yandex_io/libs/http_client/http_client.h>
#include <yandex_io/libs/ipc/i_ipc_factory.h>
#include <yandex_io/libs/threading/periodic_executor.h>
#include <yandex_io/libs/threading/steady_condition_variable.h>
#include <yandex_io/sdk/backend_config_observer.h>
#include <yandex_io/sdk/push_notification_observer.h>
#include <yandex_io/sdk/sdk_interface.h>

#include <atomic>
#include <map>
#include <memory>
#include <string>

namespace YandexIO {

    class BluetoothStreamOutManager final: public Bluetooth::BluetoothEventListener, public BackendConfigObserver, public PushNotificationObserver {
    public:
        BluetoothStreamOutManager(
            std::shared_ptr<YandexIO::IDevice> device,
            std::shared_ptr<quasar::IAuthProvider> authProvider,
            std::shared_ptr<Bluetooth> bluetoothImpl,
            std::string backendUrl,
            std::string deviceId,
            std::string platform);

        /**
         * @brief Connect to external acoustics if user set up "WantPairWithAcoustics" in User Application to true and
         * set up any acoustics connect to
         */
        void runStreamOut();
        /**
         * not implemented now
         */
        void stopStreamOut();

        void subscribeOnConfig(const std::shared_ptr<SDKInterface>& sdk);

        void onDeviceConfig(const std::string& configName, const std::string& jsonConfigValue) override;

        void onPushNotification(const std::string& operation, const std::string& messageStr) override;

        /* Remove copy constructor */
        BluetoothStreamOutManager(const BluetoothStreamOutManager&) = delete;
        BluetoothStreamOutManager& operator=(const BluetoothStreamOutManager&) = delete;

        ~BluetoothStreamOutManager();

        void onBaseEvent(Bluetooth::BaseEvent ev, const Bluetooth::EventResult& res) override;

        void onSourceEvent(Bluetooth::SourceEvent ev, const Bluetooth::EventResult& res) override;

        void onSinkEvent(Bluetooth::SinkEvent /* ev */, const Bluetooth::EventResult& /* res */) override{};

    private:
        void onScanProgress(const Bluetooth::EventResult& res);
        void onScanDone(const Bluetooth::EventResult& res);
        void sendScanResult(const Bluetooth::EventResult& res, std::stack<std::string>& scanIdsStack, bool isDone = false);

        void onSourceConnectionEvent(Bluetooth::SourceEvent /*ev*/, Bluetooth::EventResult /*res*/);

        std::shared_ptr<YandexIO::IDevice> device_;
        std::shared_ptr<quasar::IAuthProvider> authProvider_;

        std::shared_ptr<Bluetooth> bluetoothImpl_;
        /* Last scan id is the most fresh scan id, so we should send value to the most fresh scan id as fast as we can
         * So use stack to get Top value (most recent)
         */
        std::stack<std::string> scanIdsStack_;

        /**
         * mutex_ is a protection for scanIdsStack_, pairedNetworks_ and currentlyPairedMac_
         */
        std::mutex mutex_;
        quasar::HttpClient httpClient_;
        const std::string backendUrl_;
        const std::string deviceId_;
        const std::string platform_;

        /**
         * @brief isStreamOutEnabled_ is responsible for "WantPairWithAcoustics" toggle in User Application.
         *        true - if user want to connect to acoustics, false - if don't
         */
        std::atomic_bool isStreamOutEnabled_;

        std::string bluetoothPairedDevicesCache_;
        std::vector<Bluetooth::BtNetwork> pairedNetworks_;
        std::string currentlyPairedMac_;
        int currPairedIdx_;
        static constexpr int PAIR_RETRIES_MAX_COUNT = 5;
        int pairRetries_;

        /* TODO: end up state machine */
        enum class BluetoothDeviceState: int8_t {
            IDLE,
            SCANNING,
            CONNECTING,
            DISCONNECTING
        };

        enum class BluetoothDeviceEvent {
            START_CONNECTING,
            STOP_CONNECTING,
            START_DISCONNECTING,
            STOP_DISCONNECTING,
            START_SCANNING,
            STOP_SCANNING
        };

        void updateBluetoothState(BluetoothDeviceEvent ev);

        BluetoothDeviceState state_;

        TaskManager taskManager_;

        void waitIdle();

        std::mutex idleStateMutex_;
        quasar::SteadyConditionVariable idleCondVar_;

        std::thread tasksThread_;
        std::atomic_bool threadExists_;
        std::unique_ptr<quasar::PeriodicExecutor> reconnectExecutor_;

        /**
         * @brief Check is it needed to connect to external acoustics
         * @return true - if bluetooth power state is ON and isStreamOutEnabled_ == true; otherwise return false
         */
        bool shouldConnect() const;

        /**
         * @brief Perform reconnecting to external acoustics if it is needed (check shouldConnect()). The reconnection will
         * be done in 20 seconds after this call. Call is non blocking. After 20 sec delay Reconnecting thread will also
         * check if reconnect is needed (if not -> drop reconnect).
         */
        void performReconnect();

        void tasksThread();
        static Json::Value createNetworksJsonArray(const std::set<Bluetooth::BtNetwork>& networks);
    };

} // namespace YandexIO
