#include "bluetooth_tasks_manager.h"

#include <yandex_io/libs/logging/logging.h>

#include <memory>
#include <sstream>

using namespace YandexIO;

const std::map<BluetoothTask::Task, TaskManager::TaskPriority> TaskManager::taskPriorityMap{
    {BluetoothTask::Task::BLUETOOTH_ON, TaskPriority::URGENT},
    {BluetoothTask::Task::BLUETOOTH_OFF, TaskPriority::URGENT},
    {BluetoothTask::Task::SCAN_NETWORKS, TaskPriority::MAJOR},
    {BluetoothTask::Task::PAIR_WITH, TaskPriority::MINOR},
    {BluetoothTask::Task::DISCONNECT, TaskPriority::MINOR},
    {BluetoothTask::Task::RECONNECT, TaskPriority::VERY_MINOR}};

/* This function should be called after waitTask method. Otherwise it cat throw runtime_error */
BluetoothTask TaskManager::takeTask()
{
    std::lock_guard<std::mutex> guard(tasksMutex_);
    for (auto& i : tasks_) {
        if (i != nullptr) {
            auto task(std::move(i));
            i = nullptr;
            return *task;
        }
    }
    throw std::runtime_error("Empty Task Holder");
    /* User should be sure that there are any task in TaskManager (use hasTask or waitTask methods) */
}

/* Should be called with mutex */
bool TaskManager::hasTask() const {
    for (auto& i : tasks_) {
        if (i != nullptr) {
            return true;
        }
    }
    return false;
}

TaskManager::~TaskManager()
{
    /* Task manager is dying. Add task to wake up waitTask */
    BluetoothTask task;
    task.task = BluetoothTask::Task::BLUETOOTH_OFF;
    addTask(task);
}

void TaskManager::addTask(const BluetoothTask& task)
{
    std::lock_guard<std::mutex> guard(tasksMutex_);
    std::stringstream ss;
    ss << "Add: Task: " << (int)task.task << ". With prio: " << (int)taskPriorityMap.at(task.task);
    YIO_LOG_DEBUG(ss.str());

    tasks_[(int)taskPriorityMap.at(task.task)] = std::make_unique<BluetoothTask>(task);
    taskCondVar_.notify_one();
}

/* Wait until task will be added.
 * If already has task condVar will check predicate and return
 */
void TaskManager::waitTask() const {
    std::unique_lock<std::mutex> lock(tasksMutex_);
    auto predicate = [this]() {
        return hasTask();
    };
    taskCondVar_.wait(lock, std::cref(predicate));
}
