#include "bluetooth_telemetry.h"

#include <yandex_io/libs/json_utils/json_utils.h>

using namespace YandexIO;

BluetoothTelemetry::BluetoothTelemetry(std::shared_ptr<ITelemetry> telemetry, bool isBtOnAtStart, BluetoothTelemetry::TelemetrySettings settings)
    : telemetry_(std::move(telemetry))
    , settings_(settings)
{
    if (settings_.power) {
        const int isBtOn = isBtOnAtStart ? 1 : 0;
        telemetry_->putAppEnvironmentValue("is_bluetooth_on", std::to_string(isBtOn));
    }
}

void BluetoothTelemetry::onBaseEvent(Bluetooth::BaseEvent ev, const Bluetooth::EventResult& /*res*/) {
    if (!settings_.power) {
        return;
    }
    switch (ev) {
        case Bluetooth::BaseEvent::POWER_OFF: {
            telemetry_->putAppEnvironmentValue("is_bluetooth_on", "0");
            break;
        }
        case Bluetooth::BaseEvent::POWER_ON: {
            telemetry_->putAppEnvironmentValue("is_bluetooth_on", "1");
            break;
        }
        default: {
            break;
        }
    }
}

void BluetoothTelemetry::onSourceEvent(Bluetooth::SourceEvent /*ev*/, const Bluetooth::EventResult& /*res*/) {
}

void BluetoothTelemetry::onSinkEvent(Bluetooth::SinkEvent ev, const Bluetooth::EventResult& res) {
    if (res.result != Bluetooth::EventResult::Result::OK) {
        return;
    }
    switch (ev) {
        case Bluetooth::SinkEvent::CONNECTED: {
            connectedDevices_.insert(res.network.addr);
            if (settings_.connections) {
                telemetry_->reportEvent("bluetooth_source_connected");
                telemetry_->putAppEnvironmentValue("is_bt_source_connected", std::to_string(connectedDevices_.size()));
            }
            break;
        }
        case Bluetooth::SinkEvent::DISCONNECTED: {
            connectedDevices_.erase(res.network.addr);
            if (connectedDevices_.empty()) {
                executor_.reset();
            }
            if (settings_.connections) {
                telemetry_->reportEvent("bluetooth_source_disconnected");
                telemetry_->putAppEnvironmentValue("is_bt_source_connected", std::to_string(connectedDevices_.size()));
            }
            break;
        }
        case Bluetooth::SinkEvent::AVRCP_IN: {
            switch (res.avrcpEvent) {
                case Bluetooth::AVRCP::PLAY_STOP:
                case Bluetooth::AVRCP::PLAY_PAUSE: {
                    executor_.reset();
                    break;
                } /* PLAY_PAUSE */
                case Bluetooth::AVRCP::PLAY_START: {
                    if (settings_.heartbeat) {
                        executor_ = std::make_unique<quasar::PeriodicExecutor>([this]() {
                            std::scoped_lock guard(mutex_);
                            telemetry_->reportEvent("bluetoothHeartbeat", quasar::jsonToString(trackInfo_.toJson()));
                        }, settings_.heartbeatPeriod, quasar::PeriodicExecutor::PeriodicType::SLEEP_FIRST);
                    }
                    break;
                } /* PLAY_START */
                case Bluetooth::AVRCP::TRACK_META_INFO: {
                    std::scoped_lock guard(mutex_);
                    trackInfo_ = res.trackInfo;
                } /* TRACK_META_INFO */
                default: {
                    break;
                }
            }
            break;
        }
        default: {
            break;
        }
    }
}
