#include "bluetooth_android.h"

#include <yandex_io/modules/bluetooth/util/util.h>

#include <yandex_io/libs/logging/logging.h>

#include <cmath>
#include <sstream>

using namespace YandexIO;
using namespace quasar;

BluetoothAndroid::BluetoothAndroid(std::shared_ptr<ipc::IIpcFactory> ipcFactory)
    : Bluetooth(BLUETOOTH_NAME)
    , toBluetooth_(ipcFactory->createIpcConnector("bluetoothd"))
{
    toBluetooth_->setMessageHandler(std::bind(&BluetoothAndroid::handleBluetoothMessage, this, std::placeholders::_1));
    toBluetooth_->tryConnectToService();
}

void BluetoothAndroid::handleBluetoothMessage(const ipc::SharedMessage& message) {
    if (message->has_bluetooth_sink_event()) {
        auto& bluetoothSinkEvent = message->bluetooth_sink_event();
        if (bluetoothSinkEvent.has_audio_event()) {
            if (bluetoothSinkEvent.audio_event() == proto::BluetoothSinkEvent_BtSinkAudioEvent_PLAYING) {
                sinkEventCallbackLocked(SinkEvent::AVRCP_IN, EventResult{.avrcpEvent = AVRCP::PLAY_START});
            } else if (bluetoothSinkEvent.audio_event() == proto::BluetoothSinkEvent_BtSinkAudioEvent_NOT_PLAYING) {
                sinkEventCallbackLocked(SinkEvent::AVRCP_IN, EventResult{.avrcpEvent = AVRCP::PLAY_PAUSE});
            }
        }
        if (bluetoothSinkEvent.has_bluetooth_track_meta_info()) {
            sinkEventCallbackLocked(SinkEvent::AVRCP_IN, EventResult{.avrcpEvent = AVRCP::TRACK_META_INFO, .trackInfo = prepareTrackInfo(bluetoothSinkEvent.bluetooth_track_meta_info())});
        }
        if (bluetoothSinkEvent.has_connection_event()) {
            const auto& connectionEvent = bluetoothSinkEvent.connection_event();
            BtNetwork network;
            if (connectionEvent.has_network()) {
                const auto& networkInfo = connectionEvent.network();
                network.addr = networkInfo.address();
                if (networkInfo.has_name()) {
                    network.addr = networkInfo.name();
                }
                if (networkInfo.has_role()) {
                    network.role = convertRole(networkInfo.role());
                }
            }
            if (connectionEvent.connection_event() == proto::BtConnection::DISCONNECTED) {
                sinkEventCallbackLocked(SinkEvent::DISCONNECTED, EventResult{.network = std::move(network)});
            } else if (connectionEvent.connection_event() == proto::BtConnection::CONNECTED) {
                sinkEventCallbackLocked(SinkEvent::CONNECTED, EventResult{.network = std::move(network)});
            }
        }
        if (bluetoothSinkEvent.has_avrcp_event() && bluetoothSinkEvent.has_volume_percent() && bluetoothSinkEvent.avrcp_event() == proto::AVRCP::CHANGE_VOLUME_ABS) {
            auto volumePercent = bluetoothSinkEvent.volume_percent();
            if (volumePercent >= 0 && volumePercent <= 100) {
                unsigned int volumeAbs = std::round(volumePercent / 100.0 * 127);
                sinkEventCallbackLocked(SinkEvent::AVRCP_IN, EventResult{.volumeAbs = volumeAbs, .avrcpEvent = AVRCP::CHANGE_VOLUME_ABS});
            }
        }
    }
    if (message->has_bluetooth_source_event() && message->bluetooth_source_event().has_avrcp_event()) {
        const auto avrcpEvent = convertAVRCP(message->bluetooth_source_event().avrcp_event());
        sourceEventCallbackLocked(SourceEvent::AVRCP_IN, EventResult{.avrcpEvent = avrcpEvent});
    }
}

int BluetoothAndroid::scanNetworks() {
    return 0;
}

int BluetoothAndroid::powerOn() {
    return 0;
}

int BluetoothAndroid::powerOff() {
    return 0;
}

int BluetoothAndroid::stopScanNetworks() {
    return 0;
}

int BluetoothAndroid::pairWithSink(const Bluetooth::BtNetwork& /*network*/) {
    return 0;
}

int BluetoothAndroid::setVisibility(bool isDiscoverable, bool isConnectable) {
    YIO_LOG_INFO("sending visibility request");
    proto::QuasarMessage message;
    proto::BluetoothVisibility visibility = proto::BluetoothVisibility::NONE;
    if (isDiscoverable && isConnectable) {
        visibility = proto::BluetoothVisibility::CONNECTABLE_DISCOVERABLE;
    } else if (isDiscoverable) {
        visibility = proto::BluetoothVisibility::DISCOVERABLE;
    } else if (isConnectable) {
        visibility = proto::BluetoothVisibility::CONNECTABLE;
    }
    message.mutable_bluetooth_request()->mutable_sink_request()->mutable_set_visibility()->set_visibility(visibility);
    toBluetooth_->sendMessage(std::move(message));
    return 0;
}

int BluetoothAndroid::disconnectAll(BtRole /*role*/) {
    YIO_LOG_INFO("sending disconnect all request");
    proto::QuasarMessage message;
    message.mutable_bluetooth_request()->mutable_disconnect_all();
    toBluetooth_->sendMessage(std::move(message));
    return 0;
}

int BluetoothAndroid::asSinkPlayNext(const Bluetooth::BtNetwork& /*network*/) {
    YIO_LOG_INFO("sending play next request");
    proto::QuasarMessage message;
    message.mutable_bluetooth_request()->mutable_sink_request()->mutable_next();
    toBluetooth_->sendMessage(std::move(message));
    return 0;
}

int BluetoothAndroid::asSinkPlayPrev(const Bluetooth::BtNetwork& /*network*/) {
    YIO_LOG_INFO("sending play prev request");
    proto::QuasarMessage message;
    message.mutable_bluetooth_request()->mutable_sink_request()->mutable_prev();
    toBluetooth_->sendMessage(std::move(message));
    return 0;
}

int BluetoothAndroid::asSinkPlayStart(const Bluetooth::BtNetwork& /*network*/) {
    YIO_LOG_INFO("sending play start request");
    proto::QuasarMessage message;
    message.mutable_bluetooth_request()->mutable_sink_request()->mutable_play();
    toBluetooth_->sendMessage(std::move(message));
    return 0;
}

int BluetoothAndroid::asSinkPlayPause(const Bluetooth::BtNetwork& /*network*/) {
    YIO_LOG_INFO("sending play pause request");
    proto::QuasarMessage message;
    message.mutable_bluetooth_request()->mutable_sink_request()->mutable_pause();
    toBluetooth_->sendMessage(std::move(message));
    return 0;
}

int BluetoothAndroid::asSinkSetVolumeAbs(int /*volume*/) {
    // Java Bluetooth doesnt handle this event
    return 0;
}

void BluetoothAndroid::takeAudioFocus() {
    YIO_LOG_INFO("sending take audio focus request");
    proto::QuasarMessage message;
    message.mutable_media_request()->mutable_take_audio_focus();
    toBluetooth_->sendMessage(std::move(message));
}

void BluetoothAndroid::freeAudioFocus() {
    YIO_LOG_INFO("sending free audio focus request");
    proto::QuasarMessage message;
    message.mutable_media_request()->mutable_free_audio_focus();
    toBluetooth_->sendMessage(std::move(message));
}

int BluetoothAndroid::setName(const std::string& /*name*/) {
    // Java Bluetooth doesnt handle this event
    return 0;
}
