#pragma once

#include <atomic>
#include <chrono>
#include <functional>
#include <string>

class Buttons {
public:
    virtual ~Buttons();

    /**
     * @brief Main function. This function should poll button events and send them using buttons
     * callbacks
     * @param threadExists - since polling is a busy loop -> use threadExists to check that
     *                       Buttons owner still exists.
     */
    virtual void pollButtonEvents(std::atomic_bool& threadExists) = 0;

    /**
     * @brief Run pollButtonEvents thread. Should use internal threadExists flag
     */
    virtual void start() = 0;

    /**
     * @brief Stop pollButtonEvents thread (if is running)
     */
    virtual void stop() = 0;

    /**
     * @brief This function has to return value is mic muted or not
     */
    virtual bool isMicOn() const = 0;

    /**
     * @brief Button event types
     */
    enum class ButtonEvent {
        PRESS_START,  /*!< user start press on a button */
        LONG_PRESS,   /*!< user pressed on a button for a LONG_PRESS_PERIOD (should be sent once for a PRESS/RELEASE cycle */
        PRESS_RELEASE /*!< user release button in period of time < LONG_PRESS_PERIOD */
    };

    /**
     * @brief Button event handler type
     * @param buttonIndex button idx (from arch/include/defines.h ButtonsIdx)
     * @param event Button Event (PRESS_START, PRESS_RELEASE, LONG_PRESS)
     * @param timeDiff (for PRESS_RELEASE only: time difference btw PRESS_START and PRESS_RELEASE ms)
     */
    using SendButtonEventHandler = std::function<void(int buttonIndex, ButtonEvent event, std::chrono::milliseconds timeDiff)>;

    /**
     * @brief The state of microphones
     */
    enum class MicState {
        ON, /*!< Microphone is On */
        OFF /*!< Microphone is Off */
    };

    /**
     * @brief Set up callback for all buttons (except Mute (Mic) button). This callback should be called when button event happened.
     */
    void setSendButtonEventCallback(SendButtonEventHandler f) {
        sendButtonEvent_ = std::move(f);
    }

    /**
     * @brief set up long Press timeout - that should be used as a timeout for button Long Press Event
     * @param timeoutMs timeout in ms
     */
    void setLongPressTimeout(std::chrono::milliseconds timeout) {
        longPressTimeout_ = timeout;
    }

protected:
    /**
     * @brief This function should be called in pollButtonEvents function start. This function
     *        check that all necessary callbacks are set up
     * @return bool - true: can start poll. false: can't start poll
     */
    bool canStartPolling() const {
        return (sendButtonEvent_ != nullptr);
    }

    /**
     * @brief This callback should be used for all buttons
     */
    SendButtonEventHandler sendButtonEvent_;

    /**
     * @brief Timeout for Button Long Press Events
     */
    std::chrono::milliseconds longPressTimeout_ = std::chrono::milliseconds(0);
};
