#include "button_event_resolver.h"

#include <yandex_io/libs/logging/logging.h>

#include <util/system/yassert.h>

YIO_DEFINE_LOG_MODULE("buttons");

ButtonEventResolver::ButtonEventResolver(int buttonIdx,
                                         const std::shared_ptr<quasar::ICallbackQueue>& callbackQueue,
                                         std::function<void(int, Event)> callbackFunction,
                                         uint8_t maxClickLevel,
                                         std::chrono::milliseconds releaseTm,
                                         std::chrono::milliseconds nextPressTm)
    : buttonIdx_(buttonIdx)
    , callbackQueue_(callbackQueue)
    , uniqueCallback_(callbackQueue, quasar::UniqueCallback::ReplaceType::INSERT_BACK)
    , callbackFunction_(std::move(callbackFunction))
    , maxClickLevel_(maxClickLevel)
    , releaseTimeout_(releaseTm)
    , nextPressTimeout_(nextPressTm)
{
    Y_VERIFY(callbackFunction_);
    Y_VERIFY(maxClickLevel_ > 0 && maxClickLevel_ <= 3);
}

ButtonEventResolver::~ButtonEventResolver()
{
    lifetime_.die();
}

void ButtonEventResolver::buttonPressed()
{
    auto callbackQueue = callbackQueue_.lock();
    Y_ENSURE_THREAD(callbackQueue);

    if (isPressed_) {
        YIO_LOG_WARN("Button is already pressed, ignore it");
        return;
    }
    isPressed_ = true;

    resolve();
}

void ButtonEventResolver::buttonReleased()
{
    auto callbackQueue = callbackQueue_.lock();
    Y_ENSURE_THREAD(callbackQueue);

    if (!isPressed_) {
        YIO_LOG_WARN("Button is already released, ignore it");
        return;
    }
    isPressed_ = false;

    resolve();
}

void ButtonEventResolver::resolve()
{
    bool executeImmediately = false;

    if (!suggestedEvent_.has_value()) {
        suggestedEvent_ = Event::PRESSED;
    } else {
        switch (suggestedEvent_.value()) {
            case Event::RELEASED:
                suggestedEvent_ = Event::RELEASED;
                executeImmediately = true;
                break;
            case Event::PRESSED:
                suggestedEvent_ = Event::CLICK;
                if (maxClickLevel_ == 1) {
                    executeImmediately = true;
                }
                break;
            case Event::CLICK:
                suggestedEvent_ = Event::CLICK_AND_PRESSED;
                break;
            case Event::CLICK_AND_PRESSED:
                suggestedEvent_ = Event::DOUBLE_CLICK;
                if (maxClickLevel_ == 2) {
                    executeImmediately = true;
                }
                break;
            case Event::DOUBLE_CLICK:
                suggestedEvent_ = Event::DOUBLE_CLICK_AND_PRESSED;
                break;
            case Event::DOUBLE_CLICK_AND_PRESSED:
                suggestedEvent_ = Event::TRIPLE_CLICK;
                executeImmediately = true;
                break;
            default:
                YIO_LOG_ERROR_EVENT("ButtonEventResolver.ResolveFailed", "Unexpected suggestedEvent: " << (int)suggestedEvent_.value());
                suggestedEvent_.reset();
                return;
        }
    }
    YIO_LOG_DEBUG("New suggestedEvent: " << (int)suggestedEvent_.value());

    if (executeImmediately) {
        uniqueCallback_.executeImmediately([this]() {
            callbackFunction_(buttonIdx_, suggestedEvent_.value());
            suggestedEvent_.reset();
        });
    } else {
        uniqueCallback_.executeDelayed([this]() {
            callbackFunction_(buttonIdx_, suggestedEvent_.value());
            if (isPressed_) {
                suggestedEvent_ = Event::RELEASED;
            } else {
                suggestedEvent_.reset();
            }
        }, (isPressed_ ? releaseTimeout_ : nextPressTimeout_), lifetime_);
    }
}
