#include "click_manager.h"

#include <yandex_io/libs/logging/logging.h>

#include <util/system/yassert.h>

YIO_DEFINE_LOG_MODULE("buttons");

using namespace quasar;

ClickManager::ClickManager(std::set<ButtonIdx> supportedButtons,
                           std::chrono::milliseconds buttonReleaseTimeout,
                           std::chrono::milliseconds buttonNextPressTimeout)
    : buttonReleaseTimeout_(buttonReleaseTimeout)
    , buttonNextPressTimeout_(buttonNextPressTimeout)
    , callbackQueue_(std::make_shared<NamedCallbackQueue>("ClickManager"))
{
    Y_VERIFY(supportedButtons.size() <= 64);
    for (auto buttonIdx : supportedButtons) {
        buttonBitMasks_[buttonIdx] = 1 << buttonBitMasks_.size();
    }
}

ClickManager::~ClickManager()
{
    callbackQueue_->destroy();
}

bool ClickManager::isButtonSupported(ButtonIdx buttonIdx)
{
    return buttonBitMasks_.count(buttonIdx);
}

void ClickManager::buttonPressed(ButtonIdx buttonIdx)
{
    callbackQueue_->add([this, buttonIdx]() {
        if (!started_) {
            throw std::logic_error("ClickManager wasn't started!");
        }

        if (!isButtonSupported(buttonIdx)) {
            YIO_LOG_ERROR_EVENT("ClickManager.UnknownButton", "Unknown button pressed: " << buttonIdx);
            return;
        }

        auto iter = clickResolvers_.find(buttonIdx);
        if (iter != clickResolvers_.end()) {
            iter->second.buttonPressed();
        } else {
            handleButtonEvent(buttonIdx, ButtonEventResolver::Event::PRESSED);
        }
    });
}

void ClickManager::buttonReleased(ButtonIdx buttonIdx)
{
    callbackQueue_->add([this, buttonIdx]() {
        if (!started_) {
            throw std::logic_error("ClickManager wasn't started!");
        }

        if (!isButtonSupported(buttonIdx)) {
            YIO_LOG_ERROR_EVENT("ClickManager.UnknownButton", "Unknown button released: " << buttonIdx);
            return;
        }

        auto iter = clickResolvers_.find(buttonIdx);
        if (iter != clickResolvers_.end()) {
            iter->second.buttonReleased();
        } else {
            handleButtonEvent(buttonIdx, ButtonEventResolver::Event::RELEASED);
        }
    });
}

void ClickManager::handleButtonEvent(ButtonIdx buttonIdx, ButtonEventResolver::Event event)
{
    YIO_LOG_DEBUG("Handling button event: " << (int)event);
    switch (event) {
        case ButtonEventResolver::Event::CLICK_AND_PRESSED:
            handleButtonEvent(buttonIdx, ButtonEventResolver::Event::CLICK);
            handleButtonEvent(buttonIdx, ButtonEventResolver::Event::PRESSED);
            return;
        case ButtonEventResolver::Event::DOUBLE_CLICK_AND_PRESSED:
            handleButtonEvent(buttonIdx, ButtonEventResolver::Event::DOUBLE_CLICK);
            handleButtonEvent(buttonIdx, ButtonEventResolver::Event::PRESSED);
            return;
        case ButtonEventResolver::Event::PRESSED:
            onButtonPressed(buttonIdx);
            return;
        case ButtonEventResolver::Event::RELEASED:
            onButtonReleased(buttonIdx);
            return;
        case ButtonEventResolver::Event::CLICK:
        case ButtonEventResolver::Event::DOUBLE_CLICK:
        case ButtonEventResolver::Event::TRIPLE_CLICK:
            onButtonClick(buttonIdx, event);
            return;
    }
}

void ClickManager::onButtonClick(ButtonIdx buttonIdx, ButtonEventResolver::Event event)
{
    switch (currentState_) {
        case ButtonsState::IDLE:
            if (event == ButtonEventResolver::Event::CLICK) {
                auto iter = onClickActions_.find(buttonIdx);
                if (iter != onClickActions_.end()) {
                    iter->second();
                }
            } else if (event == ButtonEventResolver::Event::DOUBLE_CLICK) {
                auto iter = onDoubleClickActions_.find(buttonIdx);
                if (iter != onDoubleClickActions_.end()) {
                    iter->second();
                }
            } else if (event == ButtonEventResolver::Event::TRIPLE_CLICK) {
                auto iter = onTripleClickActions_.find(buttonIdx);
                if (iter != onTripleClickActions_.end()) {
                    iter->second();
                }
            }
            return;
        case ButtonsState::BUTTONS_PRESSED:
            currentState_ = ButtonsState::BUTTONS_RELEASING;
            deactivateButtonsCombination(currentPressedButtons_);
        case ButtonsState::BUTTONS_RELEASING:
            return;
    }
}

void ClickManager::onButtonPressed(ButtonIdx buttonIdx)
{
    previousPressedButtons_ = currentPressedButtons_;
    currentPressedButtons_ |= buttonBitMasks_[buttonIdx];
    if (previousPressedButtons_ == currentPressedButtons_) {
        YIO_LOG_ERROR_EVENT("ClickManager.AlreadyPressed", "Button is already pressed: " << buttonIdx);
        return;
    }

    switch (currentState_) {
        case ButtonsState::IDLE:
            currentState_ = ButtonsState::BUTTONS_PRESSED;
            activateButtonsCombination(currentPressedButtons_);
            return;
        case ButtonsState::BUTTONS_PRESSED:
            deactivateButtonsCombination(previousPressedButtons_);
            activateButtonsCombination(currentPressedButtons_);
            return;
        case ButtonsState::BUTTONS_RELEASING:
            return;
    }
}

void ClickManager::onButtonReleased(ButtonIdx buttonIdx)
{
    previousPressedButtons_ = currentPressedButtons_;
    currentPressedButtons_ &= ~(buttonBitMasks_[buttonIdx]);
    if (previousPressedButtons_ == currentPressedButtons_) {
        YIO_LOG_ERROR_EVENT("ClickManager.AlreadyReleased", "Button is already released: " << buttonIdx);
        return;
    }

    if (currentState_ == ButtonsState::BUTTONS_PRESSED) {
        currentState_ = ButtonsState::BUTTONS_RELEASING;
        deactivateButtonsCombination(previousPressedButtons_);
    }
    if (!currentPressedButtons_) {
        currentState_ = ButtonsState::IDLE;
        if (!wasAnyCombExecuted_) {
            YIO_LOG_INFO("No combination was executed. Trigger click instead. ButtonIdx: " << buttonIdx);
            onButtonClick(buttonIdx, ButtonEventResolver::Event::CLICK);
        }
        wasAnyCombExecuted_ = false;
    }
}

void ClickManager::activateButtonsCombination(ButtonCombMask buttonCombMask)
{
    YIO_LOG_DEBUG("Activating combinations for mask: " << buttonCombMask);
    const auto iter = buttonCombinations_.find(buttonCombMask);
    if (iter != buttonCombinations_.end()) {
        for (auto& comb : iter->second) {
            comb.activate();
        }
    }
}

void ClickManager::deactivateButtonsCombination(ButtonCombMask buttonCombMask)
{
    YIO_LOG_DEBUG("Deactivating combinations for mask: " << buttonCombMask);
    const auto iter = buttonCombinations_.find(buttonCombMask);
    if (iter != buttonCombinations_.end()) {
        for (auto& comb : iter->second) {
            comb.deactivate();
        }
    }
}

void ClickManager::addOnClickAction(ButtonIdx buttonIdx, std::function<void()> onClick)
{
    callbackQueue_->add([this, buttonIdx, onClick = std::move(onClick)]() mutable {
        onClickActions_[buttonIdx] = std::move(onClick);
    });
}

void ClickManager::addOnDoubleClickAction(ButtonIdx buttonIdx, std::function<void()> onDoubleClick)
{
    callbackQueue_->add([this, buttonIdx, onDoubleClick = std::move(onDoubleClick)]() mutable {
        onDoubleClickActions_[buttonIdx] = std::move(onDoubleClick);
    });
}

void ClickManager::addOnTripleClickAction(ButtonIdx buttonIdx, std::function<void()> onTripleClick)
{
    callbackQueue_->add([this, buttonIdx, onTripleClick = std::move(onTripleClick)]() mutable {
        onTripleClickActions_[buttonIdx] = std::move(onTripleClick);
    });
}

void ClickManager::addOnPressedAction(ButtonIdx buttonIdx, std::function<void()> onPressed,
                                      std::chrono::milliseconds pressPeriod, std::function<void()> onReleased)
{
    addOnCombinationPressedAction({buttonIdx}, std::move(onPressed), pressPeriod, std::move(onReleased));
}

void ClickManager::addOnCombinationPressedAction(std::vector<ButtonIdx> buttonIdxs, std::function<void()> onPressed,
                                                 std::chrono::milliseconds pressPeriod, std::function<void()> onReleased)
{
    callbackQueue_->add([this, buttonIdxs = std::move(buttonIdxs), onPressed = std::move(onPressed),
                         pressPeriod, onReleased = std::move(onReleased)]() mutable {
        ButtonCombMask buttonCombMask = 0;
        for (auto buttonIdx : buttonIdxs) {
            buttonCombMask |= buttonBitMasks_[buttonIdx];
        }

        buttonCombinations_[buttonCombMask].emplace_back(*this, buttonCombMask, std::move(onPressed),
                                                         pressPeriod, std::move(onReleased));
    });
}

void ClickManager::startHandlingEvents()
{
    callbackQueue_->add([this]() {
        for (auto iter : buttonBitMasks_) {
            ButtonIdx buttonIdx = iter.first;
            uint8_t maxClickLevel = 0;
            if (onTripleClickActions_.count(buttonIdx)) {
                maxClickLevel = 3;
            } else if (onDoubleClickActions_.count(buttonIdx)) {
                maxClickLevel = 2;
            } else if (onClickActions_.count(buttonIdx)) {
                maxClickLevel = 1;
            }

            if (maxClickLevel) {
                auto onEventTriggered = [this](ButtonIdx buttonIdx, ButtonEventResolver::Event event) {
                    handleButtonEvent(buttonIdx, event);
                };

                clickResolvers_.emplace(std::piecewise_construct,
                                        std::forward_as_tuple(buttonIdx),
                                        std::forward_as_tuple(buttonIdx, callbackQueue_, std::move(onEventTriggered),
                                                              maxClickLevel, buttonReleaseTimeout_, buttonNextPressTimeout_));
            }
        }
        started_ = true;
    });
}

// -- ButtonsCombination implementation --

ClickManager::ButtonsCombination::ButtonsCombination(ClickManager& manager,
                                                     ButtonCombMask buttonCombMask,
                                                     std::function<void()> onPressed,
                                                     std::chrono::milliseconds pressPeriod,
                                                     std::function<void()> onReleased)
    : callback_(manager.callbackQueue_)
    , manager_(manager)
    , buttonCombMask_(buttonCombMask)
    , onPressed_(std::move(onPressed))
    , pressPeriod_(pressPeriod)
    , onReleased_(std::move(onReleased))
{
}

ClickManager::ButtonsCombination::~ButtonsCombination()
{
    lifetime_.die();
}

void ClickManager::ButtonsCombination::activate()
{
    if (pressPeriod_.count() == 0) {
        executeOnPressed();
    } else {
        callback_.executeDelayed(std::bind(&ButtonsCombination::executeOnPressed, this), pressPeriod_, lifetime_);
    }
}

void ClickManager::ButtonsCombination::executeOnPressed()
{
    YIO_LOG_DEBUG("Combination executed");
    if (onPressed_) {
        onPressed_();
    }
    wasExecuted = true;
    manager_.wasAnyCombExecuted_ = true;
}

void ClickManager::ButtonsCombination::deactivate()
{
    if (callback_.isScheduled()) {
        callback_.reset();
    } else if (wasExecuted && onReleased_) {
        onReleased_();
    }
    wasExecuted = false;
}
