#pragma once

#include "button_event_resolver.h"

#include <yandex_io/libs/base/named_callback_queue.h>

#include <chrono>
#include <list>
#include <map>
#include <set>
#include <vector>

/**
 * @brief Allows to handle most of button logic
 */
class ClickManager {
public:
    using ButtonIdx = int;

    explicit ClickManager(std::set<ButtonIdx> supportedButtons,
                          std::chrono::milliseconds buttonReleaseTimeout,
                          std::chrono::milliseconds buttonNextPressTimeout);
    ~ClickManager();

public:
    /**
     * @brief Handles button click
     * @param buttonIdx  - index of the button
     * @param onReleased - action on button release
     */
    void addOnClickAction(
        ButtonIdx buttonIdx,
        std::function<void()> onClick);

    /**
     * @brief Handles button double click
     * @param buttonIdx     - index of the button
     * @param onDoubleClick - action on double click
     */
    void addOnDoubleClickAction(
        ButtonIdx buttonIdx,
        std::function<void()> onDoubleClick);

    /**
     * @brief Handles button triple click
     * @param buttonIdx     - index of the button
     * @param onTripleClick - action on triple click
     */
    void addOnTripleClickAction(
        ButtonIdx buttonIdx,
        std::function<void()> onTripleClick);

    /**
     * @brief Handles button pressed events
     * @note  For repeating actions use onPressed to start periodic execution
     * and onReleased to stop it
     * @param buttonIdx    - index of the button
     * @param onPressed    - action to start after pressPeriod
     * @param pressPeriod  - how long button should be pressed to call onPressed
     * @param onReleased   - function to execute after button is released
     */
    void addOnPressedAction(
        ButtonIdx buttonIdx,
        std::function<void()> onPressed,
        std::chrono::milliseconds pressPeriod = std::chrono::milliseconds(0),
        std::function<void()> onReleased = nullptr);

    /**
     * @brief Handles combination of pressed buttons
     * @note  For repeating actions use onPressed to start periodic execution
     * and onReleased to stop it
     * @param buttonIdxs   - indexes of the buttons
     * @param onPressed    - action to start after pressPeriod
     * @param pressPeriod  - how long button should be pressed to call onPressed
     * @param onReleased   - function to execute after any button in combination is released
     */
    void addOnCombinationPressedAction(
        std::vector<ButtonIdx> buttonIdxs,
        std::function<void()> onPressed,
        std::chrono::milliseconds pressPeriod = std::chrono::milliseconds(0),
        std::function<void()> onReleased = nullptr);

    /**
     * @brief call this after adding all supported actions
     */
    void startHandlingEvents();

public:
    /**
     * @brief call when button is pressed
     * @param buttonIdx - index of the button
     */
    void buttonPressed(ButtonIdx buttonIdx);

    /**
     * @brief call when button is released
     * @param buttonIdx - index of the button
     */
    void buttonReleased(ButtonIdx buttonIdx);

private:
    using ButtonBitMask = uint64_t;
    using ButtonCombMask = uint64_t;

    bool isButtonSupported(ButtonIdx buttonIdx);

    void handleButtonEvent(ButtonIdx buttonIdx, ButtonEventResolver::Event event);

    void onButtonClick(ButtonIdx buttonIdx, ButtonEventResolver::Event event);
    void onButtonPressed(ButtonIdx buttonIdx);
    void onButtonReleased(ButtonIdx buttonIdx);

    void activateButtonsCombination(ButtonCombMask buttonCombMask);
    void deactivateButtonsCombination(ButtonCombMask buttonCombMask);

private:
    enum class ButtonsState {
        IDLE,
        BUTTONS_PRESSED,
        BUTTONS_RELEASING
    };
    ButtonsState currentState_ = ButtonsState::IDLE;

    bool started_ = false;
    bool wasAnyCombExecuted_ = false;

    const std::chrono::milliseconds buttonReleaseTimeout_;
    const std::chrono::milliseconds buttonNextPressTimeout_;
    std::map<ButtonIdx, ButtonEventResolver> clickResolvers_;

    std::map<ButtonIdx, ButtonBitMask> buttonBitMasks_;
    ButtonCombMask currentPressedButtons_ = 0;
    ButtonCombMask previousPressedButtons_ = 0;

    class ButtonsCombination {
    public:
        ButtonsCombination(ClickManager& manager,
                           ButtonCombMask buttonCombMask,
                           std::function<void()> onPressed,
                           std::chrono::milliseconds pressPeriod,
                           std::function<void()> onReleased);
        ~ButtonsCombination();
        void activate();
        void deactivate();

    private:
        void executeOnPressed();
        void executeOnReleased();

        quasar::Lifetime lifetime_;
        quasar::UniqueCallback callback_;

        ClickManager& manager_;
        const ButtonCombMask buttonCombMask_;
        const std::function<void()> onPressed_;
        const std::chrono::milliseconds pressPeriod_;
        const std::function<void()> onReleased_;

        bool wasExecuted = false;
    };
    std::map<ButtonCombMask, std::list<ButtonsCombination>> buttonCombinations_;

    std::map<ButtonIdx, std::function<void()>> onClickActions_;
    std::map<ButtonIdx, std::function<void()>> onDoubleClickActions_;
    std::map<ButtonIdx, std::function<void()>> onTripleClickActions_;

    std::shared_ptr<quasar::NamedCallbackQueue> callbackQueue_;
};
