#include "swipe_event_resolver.h"

#include <yandex_io/libs/logging/logging.h>

#include <util/system/yassert.h>

YIO_DEFINE_LOG_MODULE("buttons");

SwipeEventResolver::SwipeEventResolver(const std::shared_ptr<quasar::ICallbackQueue>& callbackQueue,
                                       std::function<void(SwipeEventResolver::Event, std::vector<ButtonIdx>)> callbackFunction,
                                       uint8_t maxSwipeLength,
                                       std::chrono::milliseconds defaultTogglePressedThreshold,
                                       std::chrono::milliseconds defaultNextPressThreshold)
    : callbackQueue_(callbackQueue)
    , callbackFunction_(std::move(callbackFunction))
    , maxSwipeLength_(maxSwipeLength)
    , defaultTogglePressedThreshold_(defaultTogglePressedThreshold)
    , defaultNextPressThreshold_(defaultNextPressThreshold)
    , uniqueCallback_(callbackQueue, quasar::UniqueCallback::ReplaceType::INSERT_BACK)
{
    Y_VERIFY(callbackFunction_);
    Y_VERIFY(maxSwipeLength_ > 1); // consider using something else otherwise
}

SwipeEventResolver::~SwipeEventResolver()
{
    lifetime_.die();
}

void SwipeEventResolver::setButtonTimings(SwipeEventResolver::ButtonIdx buttonIdx,
                                          SwipeEventResolver::ButtonTimings timings)
{
    auto callbackQueue = callbackQueue_.lock();
    Y_ENSURE_THREAD(callbackQueue);

    buttonTimings_[buttonIdx] = timings;
}

void SwipeEventResolver::buttonPressed(SwipeEventResolver::ButtonIdx buttonIdx)
{
    auto callbackQueue = callbackQueue_.lock();
    if (!callbackQueue) {
        throw std::logic_error("Undefined callbackQueue");
    }
    Y_ENSURE_THREAD(callbackQueue);

    YIO_LOG_DEBUG("Pressed buttonIdx: " << buttonIdx);

    if (currentState_ == State::BTN_PRESSED || currentState_ == State::WAIT_FOR_BTN_RELEASE) {
        YIO_LOG_WARN("Another button is already pressed, ignore pressed buttonIdx: " << buttonIdx);
        return;
    }

    buttonsHistory_.push_back(buttonIdx);

    YIO_LOG_DEBUG("Before buttonIdx " << buttonIdx << " pressed state: " << (int)currentState_);
    resolve(buttonIdx);
}

void SwipeEventResolver::buttonReleased(SwipeEventResolver::ButtonIdx buttonIdx)
{
    auto callbackQueue = callbackQueue_.lock();
    if (!callbackQueue) {
        throw std::logic_error("Undefined callbackQueue");
    }
    Y_ENSURE_THREAD(callbackQueue);

    YIO_LOG_DEBUG("Released buttonIdx: " << buttonIdx);

    if (currentState_ == State::IDLE || currentState_ == State::BTN_RELEASED) {
        YIO_LOG_WARN("No button is currently pressed, ignore released buttonIdx: " << buttonIdx);
        return;
    }

    if (buttonsHistory_.size() == 0 || buttonIdx != buttonsHistory_.back()) {
        YIO_LOG_WARN("Button is released but wasn't pressed, buttonIdx: " << buttonIdx);
        return;
    }

    YIO_LOG_DEBUG("Before buttonIdx " << buttonIdx << " released state: " << (int)currentState_);
    resolve(buttonIdx);
}

void SwipeEventResolver::resolve(ButtonIdx buttonIdx)
{
    auto delay = std::chrono::milliseconds(0);

    switch (currentState_) {
        case State::BTN_RELEASED: {
            const auto historySize = buttonsHistory_.size();
            if (historySize > 1 && buttonsHistory_[historySize - 1] == buttonsHistory_[historySize - 2]) {
                YIO_LOG_DEBUG("Same button was pressed twice in a row, execute previously resolved event");
                auto lastPressedButton = buttonsHistory_[historySize - 1];

                uniqueCallback_.executeImmediately([this]() {
                    if (functionToExecute_) {
                        functionToExecute_();
                    }
                });

                /// clear pressed buttons history and only save the last pressed button
                buttonsHistory_.clear();
                buttonsHistory_.push_back(lastPressedButton);
            }
            [[fallthrough]];
        }
        case State::IDLE: {
            currentState_ = State::BTN_PRESSED;
            /// we don't want delayed execution when maximum swipe length is reached
            if (buttonsHistory_.size() != maxSwipeLength_) {
                delay = defaultTogglePressedThreshold_;
                if (auto iter = buttonTimings_.find(buttonIdx); iter != buttonTimings_.end()) {
                    delay = iter->second.togglePressedThreshold.value_or(delay);
                }
            }
            functionToExecute_ = [this, history = buttonsHistory_]() {
                auto event = Event::SWIPE;
                if (history.size() == 1) {
                    event = Event::PRESSED;
                }
                YIO_LOG_DEBUG("Event triggered: " << (int)event);
                callbackFunction_(event, history);
                currentState_ = State::WAIT_FOR_BTN_RELEASE;
            };
            break;
        }
        case State::WAIT_FOR_BTN_RELEASE: {
            functionToExecute_ = [this, history = buttonsHistory_]() {
                if (history.size() == 1) {
                    YIO_LOG_DEBUG("Event triggered: " << (int)Event::RELEASED);
                    callbackFunction_(Event::RELEASED, history);
                }
                toIdleState();
            };
            break;
        }
        case State::BTN_PRESSED: {
            currentState_ = State::BTN_RELEASED;
            delay = defaultNextPressThreshold_;
            if (auto iter = buttonTimings_.find(buttonIdx); iter != buttonTimings_.end()) {
                delay = iter->second.nextPressThreshold.value_or(delay);
            }
            functionToExecute_ = [this, history = buttonsHistory_]() {
                auto event = Event::SWIPE;
                if (history.size() == 1) {
                    event = Event::CLICK;
                }
                callbackFunction_(event, history);
                toIdleState();
            };
            break;
        }
    }

    if (delay.count() == 0) {
        uniqueCallback_.executeImmediately([this]() {
            functionToExecute_();
        });
    } else {
        uniqueCallback_.executeDelayed([this]() {
            functionToExecute_();
        }, delay, lifetime_);
    }
}

void SwipeEventResolver::toIdleState()
{
    currentState_ = State::IDLE;
    functionToExecute_ = nullptr;
    buttonsHistory_.clear();
    uniqueCallback_.reset();
}
