#pragma once

#include "i_button_events_handler.h"

#include <yandex_io/libs/threading/lifetime.h>
#include <yandex_io/libs/threading/unique_callback.h>

#include <chrono>
#include <optional>
#include <unordered_map>
#include <vector>

/**
 * Resolves swipes and clicks into events
 * Assumes that only one button can be pressed at the moment
 * Simultaneously pressed buttons will be ignored
 * After resolver triggers an event, all buttons need to be released before resolving new event
 */
class SwipeEventResolver: public IButtonEventsHandler {
public:
    enum class Event {
        PRESSED,
        RELEASED,
        CLICK,
        SWIPE
    };

    SwipeEventResolver(const std::shared_ptr<quasar::ICallbackQueue>& callbackQueue,
                       std::function<void(Event, std::vector<ButtonIdx>)> callbackFunction,
                       uint8_t maxSwipeLength,
                       std::chrono::milliseconds defaultTogglePressedThreshold,
                       std::chrono::milliseconds defaultNextPressThreshold);
    ~SwipeEventResolver();

    void buttonPressed(ButtonIdx buttonIdx) override;
    void buttonReleased(ButtonIdx buttonIdx) override;

    struct ButtonTimings {
        std::optional<std::chrono::milliseconds> togglePressedThreshold;
        std::optional<std::chrono::milliseconds> nextPressThreshold;
    };
    void setButtonTimings(ButtonIdx buttonIdx, ButtonTimings timings);

private:
    void resolve(ButtonIdx buttonIdx);
    void toIdleState();

private:
    enum class State {
        IDLE,
        BTN_PRESSED,          /// button is pressed and no Event has been triggered yet
        BTN_RELEASED,         /// button is released and no Event has been triggered yet
        WAIT_FOR_BTN_RELEASE, /// some Event has been triggered but some button is still pressed
    };

    std::weak_ptr<quasar::ICallbackQueue> callbackQueue_;
    const std::function<void(Event, std::vector<ButtonIdx>)> callbackFunction_;
    const uint8_t maxSwipeLength_;
    const std::chrono::milliseconds defaultTogglePressedThreshold_;
    const std::chrono::milliseconds defaultNextPressThreshold_;

    std::unordered_map<ButtonIdx, ButtonTimings> buttonTimings_;

    std::vector<ButtonIdx> buttonsHistory_;
    State currentState_ = State::IDLE;

    std::function<void()> functionToExecute_ = nullptr;
    quasar::UniqueCallback uniqueCallback_;
    quasar::Lifetime lifetime_;
};
