#include "swipe_manager.h"

#include <yandex_io/libs/logging/logging.h>

#include <util/system/yassert.h>

#include <algorithm>
#include <sstream>

YIO_DEFINE_LOG_MODULE("buttons");

using namespace quasar;

SwipeManager::SwipeManager(std::unordered_set<ButtonIdx> supportedButtons,
                           std::chrono::milliseconds defaultTogglePressedThreshold,
                           std::chrono::milliseconds defaultNextPressThreshold)
    : supportedButtons_(std::move(supportedButtons))
    , defaultTogglePressedThreshold_(defaultTogglePressedThreshold)
    , defaultNextPressThreshold_(defaultNextPressThreshold)
    , callbackQueue_(std::make_shared<NamedCallbackQueue>("SwipeManager"))
{
}

SwipeManager::~SwipeManager()
{
    callbackQueue_->destroy();
}

void SwipeManager::startHandlingEvents()
{
    callbackQueue_->add([this]() {
        auto onEventTriggered = [this](SwipeEventResolver::Event event, const std::vector<ButtonIdx>& buttonIdxs) {
            eventTriggered(event, buttonIdxs);
        };

        eventResolver_ = std::make_shared<SwipeEventResolver>(callbackQueue_,
                                                              std::move(onEventTriggered),
                                                              maxSwipeLength_,
                                                              defaultTogglePressedThreshold_,
                                                              defaultNextPressThreshold_);

        buttonEventsRegulator_ = std::make_unique<ButtonEventsRegulator>(eventResolver_);

        for (auto [buttonIdx, timings] : buttonTimings_) {
            eventResolver_->setButtonTimings(buttonIdx, timings);
        }

        started_ = true;
        YIO_LOG_INFO("Swipe manager was started");
    });
}

bool SwipeManager::isButtonSupported(ButtonIdx buttonIdx)
{
    return supportedButtons_.count(buttonIdx);
}

void SwipeManager::buttonPressed(ButtonIdx buttonIdx)
{
    callbackQueue_->add([this, buttonIdx]() {
        if (!started_) {
            throw std::logic_error("SwipeManager wasn't started!");
        }

        if (!isButtonSupported(buttonIdx)) {
            YIO_LOG_ERROR_EVENT("SwipeManager.UnknownButton", "Unknown button pressed: " << buttonIdx);
            return;
        }

        buttonEventsRegulator_->buttonPressed(buttonIdx);
    });
}

void SwipeManager::buttonReleased(ButtonIdx buttonIdx)
{
    callbackQueue_->add([this, buttonIdx]() {
        if (!started_) {
            throw std::logic_error("SwipeManager wasn't started!");
        }

        if (!isButtonSupported(buttonIdx)) {
            YIO_LOG_ERROR_EVENT("SwipeManager.UnknownButton", "Unknown button released: " << buttonIdx);
            return;
        }

        buttonEventsRegulator_->buttonReleased(buttonIdx);
    });
}

void SwipeManager::eventTriggered(SwipeEventResolver::Event event, const std::vector<ButtonIdx>& buttonIdxs)
{
    YIO_LOG_DEBUG("Handling event: " << (int)event);

    if (buttonIdxs.size() < 1) {
        YIO_LOG_ERROR_EVENT("SwipeManager.UnexpectedEvent",
                            "Event " << (int)event << " was triggered, but buttonIdxs.size() < 1. Ignore it");
        return;
    }

    switch (event) {
        case SwipeEventResolver::Event::PRESSED: {
            ButtonIdx buttonIdx = buttonIdxs[0];
            YIO_LOG_INFO("Button pressed: " << buttonIdx);
            if (auto iter = onPressedActions_.find(buttonIdx); iter != onPressedActions_.end()) {
                for (auto& action : iter->second) {
                    action.activate();
                }
            }
            break;
        }
        case SwipeEventResolver::Event::RELEASED: {
            ButtonIdx buttonIdx = buttonIdxs[0];
            YIO_LOG_INFO("Button released: " << buttonIdx);
            if (auto iter = onPressedActions_.find(buttonIdx); iter != onPressedActions_.end()) {
                for (auto& action : iter->second) {
                    action.deactivate();
                }
            }
            if (!wasAnyOnPressExecuted_) {
                YIO_LOG_INFO("No combination was executed. Trigger click instead. ButtonIdx: " << buttonIdx);
                eventTriggered(SwipeEventResolver::Event::CLICK, buttonIdxs);
            }
            wasAnyOnPressExecuted_ = false;
            break;
        }
        case SwipeEventResolver::Event::CLICK: {
            ButtonIdx buttonIdx = buttonIdxs[0];
            YIO_LOG_INFO("Button click: " << buttonIdx);
            if (auto iter = onClickActions_.find(buttonIdx); iter != onClickActions_.end()) {
                iter->second();
            }
            break;
        }
        case SwipeEventResolver::Event::SWIPE: {
            std::stringstream ss;
            for (auto ind : buttonIdxs) {
                ss << ind << " ";
            }
            YIO_LOG_INFO("Buttons swipe: " << ss.str());
            if (auto iter = onSwipeActions_.find(buttonIdxs); iter != onSwipeActions_.end()) {
                iter->second();
            }
            break;
        }
    }
}

void SwipeManager::addOnClickAction(ButtonIdx buttonIdx, std::function<void()> onClick)
{
    callbackQueue_->add([this, buttonIdx, onClick = std::move(onClick)]() mutable {
        onClickActions_[buttonIdx] = std::move(onClick);
    });
}

void SwipeManager::addOnSwipeAction(std::vector<ButtonIdx> buttonsIdxs, std::function<void()> onSwipe)
{
    callbackQueue_->add([this, buttonsIdxs = std::move(buttonsIdxs), onSwipe = std::move(onSwipe)]() mutable {
        maxSwipeLength_ = std::max(maxSwipeLength_, (int)buttonsIdxs.size());
        onSwipeActions_[buttonsIdxs] = std::move(onSwipe);
    });
}

void SwipeManager::addOnPressedAction(ButtonIdx buttonIdx, std::function<void()> onPressed, std::chrono::milliseconds pressPeriod,
                                      std::function<void()> onReleased)
{
    callbackQueue_->add([this, buttonIdx, onPressed = std::move(onPressed),
                         pressPeriod, onReleased = std::move(onReleased)]() mutable {
        onPressedActions_[buttonIdx].emplace_back(*this, std::move(onPressed), pressPeriod, std::move(onReleased));
    });
}

void SwipeManager::setButtonNextPressThreshold(SwipeManager::ButtonIdx buttonIdx,
                                               std::chrono::milliseconds nextPressThreshold)
{
    callbackQueue_->add([this, buttonIdx, nextPressThreshold]() {
        YIO_LOG_INFO("Set nextPressThreshold for buttonIdx: " << buttonIdx << ", value " << nextPressThreshold.count());
        buttonTimings_[buttonIdx].nextPressThreshold = nextPressThreshold;

        if (started_) {
            eventResolver_->setButtonTimings(buttonIdx, buttonTimings_[buttonIdx]);
        }
    });
}

void SwipeManager::setButtonTogglePressedThreshold(SwipeManager::ButtonIdx buttonIdx,
                                                   std::chrono::milliseconds togglePressedThreshold)
{
    callbackQueue_->add([this, buttonIdx, togglePressedThreshold]() {
        YIO_LOG_INFO("Set togglePressedThreshold for buttonIdx: " << buttonIdx << ", value " << togglePressedThreshold.count());
        buttonTimings_[buttonIdx].togglePressedThreshold = togglePressedThreshold;

        if (started_) {
            eventResolver_->setButtonTimings(buttonIdx, buttonTimings_[buttonIdx]);
        }
    });
}

SwipeManager::OnPressedAction::OnPressedAction(SwipeManager& manager,
                                               std::function<void()> onPressed,
                                               std::chrono::milliseconds pressPeriod,
                                               std::function<void()> onReleased)
    : callback_(manager.callbackQueue_)
    , manager_(manager)
    , onPressed_(std::move(onPressed))
    , pressPeriod_(pressPeriod)
    , onReleased_(std::move(onReleased))
{
}

SwipeManager::OnPressedAction::~OnPressedAction()
{
    lifetime_.die();
}

void SwipeManager::OnPressedAction::activate()
{
    if (pressPeriod_.count() == 0) {
        executeOnPressed();
    } else {
        callback_.executeDelayed(std::bind(&OnPressedAction::executeOnPressed, this), pressPeriod_, lifetime_);
    }
}

void SwipeManager::OnPressedAction::deactivate()
{
    if (callback_.isScheduled()) {
        callback_.reset();
    } else if (wasExecuted && onReleased_) {
        onReleased_();
    }
    wasExecuted = false;
}

void SwipeManager::OnPressedAction::executeOnPressed()
{
    YIO_LOG_DEBUG("OnPressed executed");
    if (onPressed_) {
        onPressed_();
    }
    wasExecuted = true;
    manager_.wasAnyOnPressExecuted_ = true;
}
