#pragma once

#include "button_events_regulator.h"
#include "swipe_event_resolver.h"

#include <yandex_io/libs/base/named_callback_queue.h>
#include <yandex_io/libs/threading/lifetime.h>
#include <yandex_io/libs/threading/unique_callback.h>

#include <chrono>
#include <list>
#include <map>
#include <unordered_map>
#include <unordered_set>

class SwipeManager {
public:
    using ButtonIdx = int;

    /**
     * @param supportedButtons
     * @param defaultTogglePressedThreshold - how long button should be pressed to be resolved as pressed
     * @param defaultNextPressThreshold     - how long button should be released to be resolved as end on input sequence
     */
    SwipeManager(std::unordered_set<ButtonIdx> supportedButtons,
                 std::chrono::milliseconds defaultTogglePressedThreshold,
                 std::chrono::milliseconds defaultNextPressThreshold);

    ~SwipeManager();

public:
    /**
     * @brief Handles button click
     * @param buttonIdx  - index of the button
     * @param onReleased - action on button release
     */
    void addOnClickAction(
        ButtonIdx buttonIdx,
        std::function<void()> onClick);

    /**
     * @brief Handles button pressed events
     * @note  For repeating actions use onPressed to start periodic execution
     * and onReleased to stop it
     * @param buttonIdx    - index of the button
     * @param onPressed    - action to start after pressPeriod
     * @param pressPeriod  - how long button should be pressed to call onPressed
     * @param onReleased   - function to execute after button is released
     */
    void addOnPressedAction(
        ButtonIdx buttonIdx,
        std::function<void()> onPressed,
        std::chrono::milliseconds pressPeriod = std::chrono::milliseconds(0),
        std::function<void()> onReleased = nullptr);

    /**
     * @brief Handles buttons 'swipes'
     * @param buttonsIdxs - buttons order in 'swipe'
     * @param onSwipe     - function to execute on swipe
     */
    void addOnSwipeAction(
        std::vector<ButtonIdx> buttonsIdxs,
        std::function<void()> onSwipe);

    /**
     * @brief specify press threshold for button
     * @param buttonIdx
     * @param togglePressedThreshold
     */
    void setButtonTogglePressedThreshold(ButtonIdx buttonIdx, std::chrono::milliseconds togglePressedThreshold);

    /**
     * @brief specify next press threshold for button
     * @param buttonIdx
     * @param nextPressThreshold
     */
    void setButtonNextPressThreshold(ButtonIdx buttonIdx, std::chrono::milliseconds nextPressThreshold);

    /**
     * @brief call this after adding all supported actions
     */
    void startHandlingEvents();

public:
    /**
     * @brief call when button is pressed
     * @param buttonIdx - index of the button
     */
    void buttonPressed(ButtonIdx buttonIdx);

    /**
     * @brief call when button is released
     * @param buttonIdx - index of the button
     */
    void buttonReleased(ButtonIdx buttonIdx);

private:
    bool isButtonSupported(ButtonIdx buttonIdx);

    void eventTriggered(SwipeEventResolver::Event event, const std::vector<ButtonIdx>& buttonIdxs);

private:
    bool started_ = false;
    bool wasAnyOnPressExecuted_ = false;

    std::unordered_set<ButtonIdx> supportedButtons_;
    const std::chrono::milliseconds defaultTogglePressedThreshold_;
    const std::chrono::milliseconds defaultNextPressThreshold_;

    class OnPressedAction {
    public:
        OnPressedAction(SwipeManager& manager,
                        std::function<void()> onPressed,
                        std::chrono::milliseconds pressPeriod,
                        std::function<void()> onReleased);
        ~OnPressedAction();
        void activate();
        void deactivate();

    private:
        void executeOnPressed();

        quasar::Lifetime lifetime_;
        quasar::UniqueCallback callback_;

        SwipeManager& manager_;
        const std::function<void()> onPressed_;
        const std::chrono::milliseconds pressPeriod_;
        const std::function<void()> onReleased_;

        bool wasExecuted = false;
    };
    std::unordered_map<ButtonIdx, std::list<OnPressedAction>> onPressedActions_;
    std::unordered_map<ButtonIdx, std::function<void()>> onClickActions_;
    std::map<std::vector<ButtonIdx>, std::function<void()>> onSwipeActions_;
    int maxSwipeLength_ = 2;

    std::unordered_map<ButtonIdx, SwipeEventResolver::ButtonTimings> buttonTimings_;

    std::unique_ptr<ButtonEventsRegulator> buttonEventsRegulator_;
    std::shared_ptr<SwipeEventResolver> eventResolver_;
    std::shared_ptr<quasar::NamedCallbackQueue> callbackQueue_;
};
