#include "dev_input_buttons.h"

#include <yandex_io/libs/errno/errno_exception.h>
#include <yandex_io/libs/logging/logging.h>
#include <yandex_io/libs/threading/utils.h>

#include <util/generic/scope.h>
#include <util/system/yassert.h>

#include <algorithm>
#include <chrono>
#include <cstring>
#include <exception>
#include <fstream>
#include <optional>
#include <sstream>
#include <vector>

#include <fcntl.h>
#include <poll.h>
#include <unistd.h>
#include <sys/ioctl.h>
#include <sys/stat.h>
#include <sys/types.h>

YIO_DEFINE_LOG_MODULE("buttons");

using std::chrono::duration_cast;
using std::chrono::milliseconds;
using std::chrono::steady_clock;
using namespace quasar;
using namespace YandexIO;

DevInputButtons::DevInputButtons(
    const std::vector<std::string>& devInputNodes,
    const std::map<uint16_t, int>& codeToButton,
    std::vector<ButtonInfo> readBeforePolling)
    : devInputNodes_(devInputNodes)
    , codeToButton_(codeToButton)
    , readBeforePolling_(readBeforePolling)
{
    longPressTimeout_ = std::chrono::seconds(5);

    for (const auto& button : readBeforePolling_) {
        Y_VERIFY(std::find(devInputNodes_.begin(), devInputNodes_.end(), button.inputNode));
        Y_VERIFY(codeToButton.count(button.buttonCode) == 1);
    }
}

void DevInputButtons::start()
{
    if (!threadExists_) {
        threadExists_ = true;
        pollThread_ = std::thread(&DevInputButtons::pollButtonEvents, this, std::ref(threadExists_));
    }
}

void DevInputButtons::stop()
{
    if (pollThread_.joinable()) {
        threadExists_ = false;
        wakeupFd_.signal();
        pollThread_.join();
    }
}

DevInputButtons::~DevInputButtons()
{
    stop();
}

void DevInputButtons::pollButtonEvents(std::atomic_bool& threadExists) {
    if (!canStartPolling()) {
        throw std::runtime_error("Callbacks are not set up");
    }

    setThreadSchedulingParams(SCHED_RR, 99);

    std::vector<struct pollfd> pollfds;
    pollfds.reserve(devInputNodes_.size() + 1);

    std::map<std::string, int> fileToFd;

    for (const auto& node : devInputNodes_) {
        int fd = 0;
        if ((fd = open(node.c_str(), O_RDONLY)) < 0) {
            YIO_LOG_ERROR_EVENT("DevInputButtons.OpenNodeFailed", "Can't open " << node << " : " << strError(errno));
            continue;
        }
        pollfds.push_back({fd, POLLIN | POLLPRI | POLLERR, 0});

        fileToFd[node] = fd;

        /* Log Node Info */
        int version = 0;
        int ret = ioctl(fd, EVIOCGVERSION, &version);
        if (ret == -1) {
            YIO_LOG_WARN("Ioctl fail: EVIOCGVERSION: " << strError(errno));
        } else {
            std::stringstream ss;
            ss << "Input driver version is " << (version >> 16) << '.'
               << ((version >> 8) * 0xff)
               << '.' << (version & 0xff);
            YIO_LOG_INFO(ss.str());
        }

        unsigned short id[4] = {0};
        ret = ioctl(fd, EVIOCGID, id);
        if (ret == -1) {
            YIO_LOG_WARN("Ioctl fail: EVIOCGID: " << strError(errno));
        } else {
            std::stringstream ss;
            ss << "Input device ID: bus 0x" << std::hex << id[ID_BUS] << " vendor 0x"
               << id[ID_VENDOR]
               << " product 0x" << id[ID_PRODUCT] << " version 0x" << id[ID_VERSION];

            YIO_LOG_INFO(ss.str());
        }

        char name[256] = "Unknown";
        ret = ioctl(fd, EVIOCGNAME(sizeof(name)), name);
        if (ret == -1) {
            YIO_LOG_WARN("Ioctl Fail: EVIOCGNAME: " << strError(errno));
        } else {
            YIO_LOG_DEBUG(std::string("Input Device name: ") + std::string(name));
        }
    }

    /* Added all dev nodes to pollfds vector. Need to add wakeUpFd last */
    pollfds.push_back({wakeupFd_.fd(), POLLIN | POLLPRI, 0});

    Y_DEFER {
        /* Do not close wake up fd -> DevInputButtons does not own it */
        for (auto i = pollfds.begin(); i != pollfds.end() - 1; ++i) {
            close(i->fd);
        }
    };

    /* Map with currently pressed buttons and time points (when buttons was pressed).
     * Need this timepoint to know - is it LongPress or Click Event
     */
    struct ButtonStatus {
        ButtonStatus(bool /*unused*/, steady_clock::time_point tp)
            : isLongPressSent(false)
            , pressStartTime(tp)
        {
        }
        bool isLongPressSent{false};
        steady_clock::time_point pressStartTime;
    };
    std::map<uint16_t, ButtonStatus> pressedButtons;

    struct input_event ev[64];
    std::optional<std::chrono::milliseconds> pollTimeout = std::nullopt; /* set up infinite timeout */

    // Take initial state of buttons we wish to read before polling
    for (const auto& button : readBeforePolling_) {
        int fd = fileToFd[button.inputNode];
        int buttonCode = button.buttonCode;
        int type = button.type;

        // Taken from https://www.linuxjournal.com/files/linuxjournal.com/linuxjournal/articles/064/6429/6429l10.html
        uint8_t keyB[buttonCode / 8 + 1];
        unsigned int param;
        memset(keyB, 0, sizeof(keyB));
        switch (type) {
            case EV_SW: {
                param = EVIOCGSW(sizeof(keyB));
                break;
            }
            case EV_KEY: {
                param = EVIOCGKEY(sizeof(keyB));
                break;
            }
            default: {
                throw std::runtime_error("Unknown key type: " + std::to_string(type));
            }
        }

        ioctl(fd, param, keyB);
        if (checkBit(buttonCode, keyB)) {
            ButtonStatus bs(false, std::chrono::steady_clock::now());
            pressedButtons.insert(std::make_pair(buttonCode, bs));
            /* Button is pressed, set up poll timeout to shortPollTimeout ms, so we don't need to wait until
             * user will free the button and send LONG_PRESS event early (when button will be pressed for 5 sec).
             */
            constexpr auto shortPollTimeoutMs = std::chrono::milliseconds(300);
            pollTimeout = shortPollTimeoutMs;
            sendButtonEvent_(codeToButton_.at(buttonCode), ButtonEvent::PRESS_START, std::chrono::milliseconds(0));

            YIO_LOG_INFO("Button " << buttonCode << " is pressed from very start");
        } else {
            sendButtonEvent_(codeToButton_.at(buttonCode), ButtonEvent::PRESS_RELEASE, std::chrono::milliseconds(0));
            YIO_LOG_INFO("Button " << buttonCode << " is released from very start");
        }
    }

    while (threadExists) {
        /* Poll /dev/input/event nodes and WakeUpFd for event */
        int ret = poll(pollfds.data(), pollfds.size(), pollTimeout ? pollTimeout->count() : -1);
        if (ret < 0) {
            YIO_LOG_ERROR_EVENT("DevInputButtons.PollFailed", "Poll failed: " << strError(errno));
            continue;
        }
        /* Wake up fd is always last */
        if (pollfds[pollfds.size() - 1].revents & (POLLIN | POLLPRI)) {
            /* Got Wake Up Fd event. It means that ButtonsEndpoint is being destroyed. so stop
             * poll buttons.
             */
            break;
        }

        if (ret == 0 /* timeout */) {
            if (!pollTimeout && pressedButtons.empty()) {
                /* When !pollTimeout it means that we do not any button pressed
                 * So sleep by next poll (wait next button event)
                 */
                continue;
            } else {
                const auto now = steady_clock::now();
                for (auto& it : pressedButtons) {
                    auto timeDiff = duration_cast<milliseconds>(now - it.second.pressStartTime);
                    if (!it.second.isLongPressSent && timeDiff > longPressTimeout_) {
                        /* During longPressTimeoutMs user did not released button */
                        const int buttonIdx = static_cast<int>(codeToButton_.at(it.first));
                        const ButtonEvent event = ButtonEvent::LONG_PRESS;
                        it.second.isLongPressSent = true; /* Long press event should be sent once for a button (in
                                                             press/release cycle */
                        sendButtonEvent_(buttonIdx, event, timeDiff);
                    }
                }
                continue;
            }
        }

        /* poll returns amount of fd's with not empty revents. Handle all of them */
        for (int j = 0; j < ret; ++j) {
            int fd = -1;
            for (auto& pfd : pollfds) {
                if (pfd.revents & (POLLIN | POLLPRI)) {
                    fd = pfd.fd;
                    pfd.revents = 0;
                    break;
                } else if (pfd.revents != 0) {
                    YIO_LOG_WARN("Error Poll Event");
                }
            }
            if (fd < 0) {
                /* Got error event -> ignore it */
                continue;
            }

            ssize_t rd = read(fd, ev, sizeof(struct input_event) * 64);
            if (rd < int(sizeof(struct input_event))) {
                YIO_LOG_ERROR_EVENT("DevInputButtons.ReadEventFailed", "Can't Read event from " << strError(errno));
                continue;
            }

            const auto now = steady_clock::now();
            /* Check currently buttons for LONG_PRESS */
            for (auto& it : pressedButtons) {
                /* This cycle is used to send LONG_PRESS event for currently pressed buttons. NOTE: it isn't a
                 * duplicate of same cycle above (in case of timeout). This cycle will send LONG_PRESS event in case if
                 * user for example press one button and at the same time click (few times) on another one. So we get
                 * button events and Poll Timeout wouldn't happen. So to prevent this situation check all pressed
                 * buttons for LONG_PRESS on each button event
                 */
                auto timeDiff = duration_cast<milliseconds>(now - it.second.pressStartTime);
                if (!it.second.isLongPressSent && timeDiff > longPressTimeout_) {
                    const int buttonIdx = static_cast<int>(codeToButton_.at(it.first));
                    const ButtonEvent event = ButtonEvent::LONG_PRESS;
                    it.second.isLongPressSent = true;
                    sendButtonEvent_(buttonIdx, event, timeDiff);
                }
            }
            /* Event polling example got from: https://elinux.org/images/9/93/Evtest.c */
            for (int i = 0; i < int(rd / sizeof(struct input_event)); ++i) {
                if (ev[i].type != EV_KEY && ev[i].type != EV_SW) {
                    /* Wait KEY and EV_SW Event only */
                    continue;
                }
                /* Check that User wait for this button. otherwise skip it */
                if (!codeToButton_.count(ev[i].code)) {
                    YIO_LOG_WARN("Unexpected button event. Button code:" << ev[i].code << ". State: " << ev[i].value);
                    continue;
                }
                if (ev[i].value == 1 /* pressed */) {
                    ButtonStatus bs(false, now);
                    pressedButtons.insert(std::make_pair(ev[i].code, bs));
                    /* Button is pressed, set up poll timeout to shortPollTimeout ms, so we don't need to wait until
                     * user will free the button and send LONG_PRESS event early (when button will be pressed for 5 sec).
                     */
                    constexpr auto shortPollTimeoutMs = std::chrono::milliseconds(300);
                    pollTimeout = shortPollTimeoutMs;
                    sendButtonEvent_(codeToButton_.at(ev[i].code), ButtonEvent::PRESS_START, std::chrono::milliseconds(0));
                } else if (ev[i].value == 0 /* not pressed */) {
                    const int buttonIdx = static_cast<int>(codeToButton_.at(ev[i].code));
                    const auto& it = pressedButtons.find(ev[i].code);

                    auto timeDiff = std::chrono::milliseconds(0);
                    if (it != pressedButtons.end()) {
                        timeDiff = duration_cast<milliseconds>(now - it->second.pressStartTime);
                        pressedButtons.erase(it);
                        /* NOTE: LONG_PRESS event should be sent before checking current button /dev/input/event events */
                    } else {
                        YIO_LOG_ERROR_EVENT("DevInputButtons.UnexpectedButtonRelease", "Unexpected Button Release event (without PRESS_START)");
                    }

                    if (pressedButtons.empty()) {
                        /* all buttons are free, set up long poll timeout (infinite) till next button event */
                        pollTimeout = std::nullopt;
                    }
                    ButtonEvent event = ButtonEvent::PRESS_RELEASE;
                    sendButtonEvent_(buttonIdx, event, timeDiff);
                } else if (ev[i].value == 2 /* autorepeat */) {
                    /* do nothing. AutoRepeat notify us that button is still pressed */
                }
            } /* for */
        }     /* for */

    } /* while */
}
