#pragma once

#include <yandex_io/modules/buttons/buttons.h>

#include <yandex_io/libs/threading/wakeup_fd.h>

#include <linux/input.h>

#include <atomic>
#include <cstdint>
#include <map>
#include <string>
#include <thread>
#include <vector>

namespace YandexIO {

    class DevInputButtons: public Buttons {
    public:
        struct ButtonInfo {
            ButtonInfo(std::string inputNode, uint16_t buttonCode, int keyType = EV_KEY)
                : inputNode(std::move(inputNode))
                , buttonCode(buttonCode)
                , type(keyType)
            {
            }

            std::string inputNode;
            uint16_t buttonCode;
            int type; // EV_SW, EV_KEY
        };

        /**
         * @param[in] devInputNodes Array with paths to /dev/input/eventX nodes
         * @param[in] codeToButton Map where key is Button Code in /dev/input/eventX (struct input_event::code)
         * and the second value is user-defined button index.
         * NOTE: ALL CODES SHOULD BE UNIQUE: If different /dev/input/eventX nodes use same code ->
         * DevInputButtons won't work for it!
         * @param[in] readBeforePolling Array of buttons which have to be read before start polling
         * NOTE: input node in every provided ButtonInfo structure must be also provided in devInputNodes,
         * button code in every provided ButtonInfo structure must be provided as codeToButton key
         */
        DevInputButtons(
            const std::vector<std::string>& devInputNodes,
            const std::map<uint16_t, int>& codeToButton,
            std::vector<ButtonInfo> readBeforePolling = std::vector<ButtonInfo>());

        virtual ~DevInputButtons();

        void start() override;

        void stop() override;

        /**
         * @brief pollButtonEvents This function poll all /dev/input/evenetX nodes (that were passed in
         * constructor). KEY events from those dev nodes are sending through sendButtonEvent_ and sendMuteButtonEvent_
         * callbacks. This function ends when WakeUpFd is signaling (it should be set up through setWakeUpFd
         * method) and threadExists is set up to false
         * @param threadExists When this bool is true pollButtonEvents poll events. Otherwise stop polling
         */
        void pollButtonEvents(std::atomic_bool& threadExists) override;

        /**
         * @NOTE: DevInputButtons do not handle mics state
         */
        bool isMicOn() const override {
            return false;
        }

    private:
        static bool checkBit(int position, const uint8_t* values) {
            int numInt = position / 8;
            int insideInt = position % 8;
            return values[numInt] & (1 << insideInt);
        }

        /**
         * @brief vector with path to /dev/input/eventX
         */
        const std::vector<std::string> devInputNodes_;

        /**
         * @brief Map where key is Button Code in /dev/input/eventX (struct input_event::code) and
         * the second value is user-defined button index
         */
        const std::map<uint16_t, int> codeToButton_;

        const std::vector<ButtonInfo> readBeforePolling_;

        quasar::WakeupFd wakeupFd_;
        std::atomic_bool threadExists_{false};
        std::thread pollThread_;
    };

} /* namespace YandexIO */
