#include "clock_display_state_holder.h"

#include <yandex_io/libs/base/persistent_file.h>
#include <yandex_io/libs/base/utils.h>
#include <yandex_io/libs/json_utils/json_utils.h>
#include <yandex_io/libs/logging/logging.h>

YIO_DEFINE_LOG_MODULE("clock_display_state_holder");

ClockDisplayStateHolder::ClockDisplayStateHolder(TFsPath stateFilePath, std::shared_ptr<YandexIO::SDKInterface> sdk)
    : stateFilePath_(std::move(stateFilePath))
    , sdk_(std::move(sdk))
{
    if (stateFilePath_.Exists()) {
        try {
            const auto value = quasar::readJsonFromFile(stateFilePath_.GetPath());
            state_.isClockOn = quasar::tryGetBool(value, "isClockOn", state_.isClockOn);
            state_.clockEnabled = quasar::tryGetBool(value, "clockEnabled", state_.clockEnabled);
            state_.showClockDuringDnd = quasar::tryGetBool(value, "showClockDuringDnd", state_.showClockDuringDnd);
            state_.brightness = quasar::tryGetDouble(value, "brightness", state_.brightness);
        } catch (const std::exception& e) {
            YIO_LOG_ERROR_EVENT("ClockDisplayStateHolder.CantLoadSavedState", "Error while loading saved state: " << e.what());
        }
    }
    sdk_->reportClockDisplayState(state_);
}

void ClockDisplayStateHolder::addListener(std::weak_ptr<Listener> wlistener) {
    std::scoped_lock<std::mutex> lock(mutex_);
    listeners_.push_back(wlistener);
    if (auto listener = wlistener.lock()) {
        listener->onClockDisplayStateChanged(state_);
    }
}

void ClockDisplayStateHolder::setBrightness(double brightness) {
    std::scoped_lock<std::mutex> lock(mutex_);
    if (state_.brightness == brightness) {
        return;
    }

    state_.brightness = brightness;
    saveStateAndReport(false);
}

void ClockDisplayStateHolder::setClockBehaviour(bool isClockOn, bool enabled, bool showClockDuringDnd) {
    std::scoped_lock<std::mutex> lock(mutex_);
    if (state_.isClockOn != isClockOn || state_.clockEnabled != enabled || state_.showClockDuringDnd != showClockDuringDnd) {
        state_.isClockOn = isClockOn;
        state_.clockEnabled = enabled;
        state_.showClockDuringDnd = showClockDuringDnd;
        saveStateAndReport(true);
    }
}

YandexIO::ClockDisplayState ClockDisplayStateHolder::getClockDisplayState() const {
    std::scoped_lock<std::mutex> lock(mutex_);
    return state_;
}

void ClockDisplayStateHolder::saveStateAndReport(bool shouldLog) const {
    saveState(shouldLog);

    sdk_->reportClockDisplayState(state_);
    for (const auto& wlistener : listeners_) {
        if (auto listener = wlistener.lock()) {
            listener->onClockDisplayStateChanged(state_);
        }
    }
}

void ClockDisplayStateHolder::saveState(bool shouldLog) const {
    Json::Value json;
    json["isClockOn"] = state_.isClockOn;
    json["clockEnabled"] = state_.clockEnabled;
    json["showClockDuringDnd"] = state_.showClockDuringDnd;
    json["brightness"] = state_.brightness;

    std::string toString = quasar::jsonToString(json);

    quasar::PersistentFile settingsFile(stateFilePath_.GetPath(), quasar::PersistentFile::Mode::TRUNCATE);
    settingsFile.write(toString);
    if (shouldLog) {
        YIO_LOG_INFO("Saved ClockDisplayState: " << toString);
    } else {
        YIO_LOG_DEBUG("Saved ClockDisplayState: " << toString);
    }
}
