#include "data_resetter.h"

#include <yandex_io/modules/sync_file_player_wrapper/sync_file_player_wrapper.h>
#include <yandex_io/libs/logging/logging.h>

#include <util/generic/scope.h>

namespace YandexIO {

    DataResetter::DataResetter(std::shared_ptr<SDKInterface> sdk,
                               std::shared_ptr<DataResetExecutor> executor,
                               DataResetterSettings dataResetterSettings)
        : dataResetExecutor_(std::move(executor))
        , sdk_(std::move(sdk))
        , voiceAssistantBlocker_(sdk_, "data_reset")
        , settings_(std::move(dataResetterSettings))
    {
        isScheduled_ = false;
        scheduleCanceled_ = true;
        waitConfirm_ = false;
        dataResetThread_ = std::thread(&DataResetter::dataResetThread, this);
    }

    DataResetter::~DataResetter()
    {
        {
            std::scoped_lock guard(mutex_);
            stopped_ = true;
            condVar_.notify_one();
        }
        dataResetThread_.join();
    }

    void DataResetter::schedule()
    {
        std::lock_guard<std::mutex> guard(mutex_);
        /* Schedule data reset */
        scheduleCanceled_ = false;
        condVar_.notify_one();
    }

    // FIXME: Use settings for Timeout and SoundFiles
    void DataResetter::dataResetThread()
    {
        while (!stopped_) {
            /* Wait Data Reset to start */
            const bool startDataReset = waitDataResetStart();
            if (!startDataReset) {
                /* waitDataResetStart was canceled by destructor */
                return;
            }

            /* Wait until user confirm Data reset */
            notifyWaitConfirm();
            const auto waitConfirmResult = waitConfirm();
            switch (waitConfirmResult) {
                case WaitConfirmResult::STOPPED: {
                    /* destructor is called. stop execution */
                    voiceAssistantBlocker_.unblock();
                    notifyCanceled();
                    return;
                }
                case WaitConfirmResult::CONFIRMED: {
                    /* Notify YandexIOSDK that Data Reset was confirmed */
                    YIO_LOG_INFO("Data Reset was confirmed. execute Data Reset");
                    notifyExecuting();
                    syncPlaySoundFile(sdk_->getFilePlayerCapability(),
                                      settings_.successDataResetTrack,
                                      quasar::proto::AudioChannel::DIALOG_CHANNEL,
                                      std::chrono::seconds(10));
                    dataResetExecutor_->execute();
                    break;
                }
                case WaitConfirmResult::CONFIRM_TIMEOUT: {
                    YIO_LOG_INFO("Data Reset Canceled by timeout");
                    notifyCanceled();
                    sdk_->getFilePlayerCapability()->playSoundFile(settings_.cancelDataResetTrack, quasar::proto::AudioChannel::DIALOG_CHANNEL);
                    break;
                }
            }
            voiceAssistantBlocker_.unblock();
        }
    }

    void DataResetter::cancelSchedule()
    {
        std::lock_guard<std::mutex> guard(mutex_);
        scheduleCanceled_ = true;
        condVar_.notify_one();
    }

    bool DataResetter::isWaitingConfirm() const {
        std::lock_guard<std::mutex> guard(mutex_);
        return waitConfirm_;
    }

    bool DataResetter::isScheduled() const {
        std::lock_guard<std::mutex> guard(mutex_);
        return isScheduled_;
    }

    DataResetter::WaitConfirmResult DataResetter::waitConfirm()
    {
        /* Notify Quasar that Data Reset started */
        sdk_->stopSetupMode();
        voiceAssistantBlocker_.block();
        std::unique_lock<std::mutex> lock(mutex_);
        confirmed_ = false;
        waitConfirm_ = true;
        /* Notify user that Data reset is waiting for confirmation */
        sdk_->getFilePlayerCapability()->playSoundFile(settings_.confirmDataResetTrack, quasar::proto::AudioChannel::DIALOG_CHANNEL);
        condVar_.wait_for(lock, settings_.waitForConfirmDataResetTime, [this]() {
            return confirmed_ || stopped_;
        });
        waitConfirm_ = false;
        if (stopped_) {
            /* interrupted by destructor */
            return WaitConfirmResult::STOPPED;
        }
        if (confirmed_) {
            /* User confirmed Data Reset */
            return WaitConfirmResult::CONFIRMED;
        }
        return WaitConfirmResult::CONFIRM_TIMEOUT;
    }

    bool DataResetter::waitDataResetStart()
    {
        std::unique_lock<std::mutex> lock(mutex_);
        while (!stopped_) {
            Y_DEFER {
                scheduleCanceled_ = true;
                waitConfirm_ = false;
            };
            /* Wait Data Reset to be scheduled */
            condVar_.wait(lock, [this]() {
                return !scheduleCanceled_ || stopped_;
            });
            if (stopped_) {
                /* Destructor is called. Stop waiting loop */
                return false;
            }
            isScheduled_ = true; /* Data reset is "scheduled" only during waitForCancelScheduleTime */
            /* Wait until data reset schedule will be canceled */
            const bool canceledStart = condVar_.wait_for(lock, settings_.waitForCancelScheduleTime, [this]() {
                return scheduleCanceled_ || stopped_;
            });
            isScheduled_ = false;
            if (canceledStart) {
                /* Data reset schedule was canceled or destructor is called. Go to next iteration */
                continue;
            }
            /* Data Reset wasn't canceled. Go to Confirmation stage */
            return true;
        }
        /* Data reset thread was canceled by destructor */
        return false;
    }

    void DataResetter::confirmDataReset()
    {
        YIO_LOG_DEBUG("Confirm Data Reset");
        std::lock_guard<std::mutex> guard(mutex_);
        confirmed_ = true;
        condVar_.notify_one();
    }

    void DataResetter::addListener(std::weak_ptr<IDataResetStateListener> listener) {
        std::scoped_lock<std::mutex> guard(mutex_);
        listeners_.push_back(std::move(listener));
    }

    void DataResetter::notifyWaitConfirm() {
        const auto currentListeners = getListeners();
        for (const auto& stateListener : currentListeners) {
            if (auto listener = stateListener.lock()) {
                listener->onDataResetWaitConfirm();
            }
        }
    }

    void DataResetter::notifyExecuting() {
        const auto currentListeners = getListeners();
        for (const auto& stateListener : currentListeners) {
            if (auto listener = stateListener.lock()) {
                listener->onDataResetExecuting();
            }
        }
    }

    void DataResetter::notifyCanceled() {
        const auto currentListeners = getListeners();
        for (const auto& stateListener : currentListeners) {
            if (auto listener = stateListener.lock()) {
                listener->onDataResetCanceled();
            }
        }
    }

    std::list<std::weak_ptr<IDataResetStateListener>> DataResetter::getListeners() const {
        std::scoped_lock<std::mutex> guard(mutex_);
        return listeners_;
    }

} // namespace YandexIO
