#pragma once

#include "data_reset_executor.h"
#include "i_data_reset_state_listener.h"

#include <yandex_io/modules/voice_assistant_blocker/voice_assistant_blocker.h>

#include <yandex_io/libs/threading/steady_condition_variable.h>

#include <atomic>
#include <list>
#include <memory>
#include <mutex>
#include <thread>

/**
 * @brief DataResetter implement DataReset feature. DataResetter have a thread that wait until Data Reset
 *        will be scheduled by "schedule" method. After schedule DataResetter wait for "schedule timeout".
 *        if schedule wasn't canceled by "cancelSchedule" method Data Reset will start wait for the Data Reset
 *        Confirmation. If Data Reset was confirmed DataResetter will play "DataResetConfirmed" and after that
 *        execute Data Reset.
 * @NOTE: "DataResetConfirmed" sound will be played synchronously and only AFTER that DataResetExecutor::execute
 *        method will be called
 */

namespace YandexIO {

    class DataResetter {
    public:
        struct DataResetterSettings {
            DataResetterSettings(){};
            DataResetterSettings(std::string confirmDataResetTrack,
                                 std::string cancelDataResetTrack,
                                 std::string successDataResetTrack,
                                 std::chrono::seconds waitForCancelScheduleTime,
                                 std::chrono::seconds waitForConfirmDataResetTime)
                : confirmDataResetTrack(std::move(confirmDataResetTrack))
                , cancelDataResetTrack(std::move(cancelDataResetTrack))
                , successDataResetTrack(std::move(successDataResetTrack))
                , waitForCancelScheduleTime(waitForCancelScheduleTime)
                , waitForConfirmDataResetTime(waitForConfirmDataResetTime)
            {
            }

            std::string confirmDataResetTrack{"confirm_data_reset.mp3"};
            std::string cancelDataResetTrack{"data_reset_canceled.mp3"};
            std::string successDataResetTrack{"data_reset_success.mp3"};
            std::chrono::seconds waitForCancelScheduleTime{10};
            std::chrono::seconds waitForConfirmDataResetTime{30};
        };

        DataResetter(std::shared_ptr<SDKInterface> sdk,
                     std::shared_ptr<DataResetExecutor> executor,
                     DataResetterSettings dataResetterSettings = DataResetterSettings());
        ~DataResetter();

        /**
         * @brief Schedule Data reset. Schedule can be canceled by cancelSchedule method. If data reset won't
         *        be canceled in timeout from settings DataResetter will wait for confirm
         */
        void schedule();

        /**
         * @brief Reset thread that will start Data Reset to state when it waits for schedule.
         *        If data reset is already started (and it's waiting for confirm) this method
         *        won't cancel currently running Data Reset
         */
        void cancelSchedule();

        /**
         * @return true - If Data Reset was "scheduled". If DataResetter is waiting for confirm -> it's not scheduled
         *                schedule means -> "will be fired (will start wait for data reset) in waitForCancelScheduleTime"
         */
        bool isScheduled() const;

        /**
         * @return true - Device is in Data Reset mode (waiting for confirm)
         */
        bool isWaitingConfirm() const;

        /**
         * @brief If DataResetter is waiting for Confirm -> it will execute data reset using DataResetExecutor
         */
        void confirmDataReset();

        /**
         * @brief Add DataResetStateListener.
         */
        void addListener(std::weak_ptr<IDataResetStateListener> listener);

    private:
        void dataResetThread();

        /**
         * @brief busy loop that wait until Data Reset is scheduled by user.
         * @return true - Data Reset is scheduled. false - busy loop was canceled by destructor
         */
        bool waitDataResetStart();

        enum class WaitConfirmResult {
            STOPPED,        /* Canceled by destructor */
            CONFIRMED,      /* Confirmed by user */
            CONFIRM_TIMEOUT /* User did not confirm data reset in timeout */
        };
        /**
         * @brief Notify user that device is waiting for data reset confirm and wait (with timeout)
         *        until confirmDataReset method is called
         */
        WaitConfirmResult waitConfirm();

        /**
         * @brief Notify listeners about waiting confirmation.
         */
        void notifyWaitConfirm();

        /**
         * @brief Notify listeners about data reset executing.
         */
        void notifyExecuting();

        /**
         * @brief Notify listeners if data reset was canceled.
         */
        void notifyCanceled();

        /**
         * @return list of listeners
         */
        std::list<std::weak_ptr<IDataResetStateListener>> getListeners() const;

        std::atomic_bool stopped_{false};
        std::thread dataResetThread_;

        mutable std::mutex mutex_;
        quasar::SteadyConditionVariable condVar_;
        bool isScheduled_{false};
        bool scheduleCanceled_{true};
        bool confirmed_{false};
        bool waitConfirm_{false};

        std::shared_ptr<DataResetExecutor> dataResetExecutor_;

        std::list<std::weak_ptr<IDataResetStateListener>> listeners_;

        const std::shared_ptr<SDKInterface> sdk_;
        VoiceAssistantBlocker voiceAssistantBlocker_;

        const DataResetterSettings settings_;
    };

} // namespace YandexIO
