#include "demo_config_manager.h"

#include <yandex_io/libs/base/persistent_file.h>
#include <yandex_io/libs/base/utils.h>
#include <yandex_io/libs/http_client/http_client.h>
#include <yandex_io/libs/json_utils/json_utils.h>
#include <yandex_io/libs/logging/logging.h>

#include <deque>
#include <map>
#include <vector>

YIO_DEFINE_LOG_MODULE("demo_mode");

using namespace quasar;

DemoConfigManager::DemoConfigManager(const Json::Value& config, std::shared_ptr<YandexIO::IDevice> device, std::shared_ptr<IDemoProvider> module)
    : device_(std::move(device))
    , cachePath_(getString(config, "demoModeCachePath"))
    , configPath_(tryGetString(config, "demoModeConfigPath"))
    , httpClientTimeoutMs_(tryGetInt(config, "httpClientTimeoutMs", DEFAULT_HTTP_TIMEOUT_MS))
    , httpClientRetriesCount_(tryGetInt(config, "httpClientRetriesCount", DEFAULT_HTTP_RETRIES))
    , module_(std::move(module))
{
    if (config.isMember("pauseBetweenStoriesMs")) {
        pauseBetweenItemsMs_ = std::make_optional<std::chrono::milliseconds>(getInt(config, "pauseBetweenStoriesMs"));
    }
}

void DemoConfigManager::prepareItems() {
    std::lock_guard<std::mutex> lock(mutex_);
    try {
        if (fileExists(configPath_)) {
            const auto fileContent = getFileContent(configPath_);
            module_->updateDemoItems(parseJson(fileContent));
            if (!checkCachedFiles(module_)) {
                YIO_LOG_WARN("Can't verify cached files");
                cleanCacheDirectory();
            }
        }
        if (!checkCachedFiles(module_)) {
            cleanCacheDirectory();
            downloadDemoFiles(module_);
        }
    } catch (const Json::Exception& e) {
        YIO_LOG_ERROR_EVENT("DemoConfigManager.PrepareItems.Exception", "Can't start demo at " << configPath_);
        notifyDemoModeError();
    }
}

void DemoConfigManager::onSystemConfig(const std::string& configName, const std::string& jsonConfigValue) {
    if (configName == DEMO_MODE_CONFIG) {
        const auto config = tryParseJson(jsonConfigValue);
        if (config.has_value() && !config->isNull()) {
            YIO_LOG_INFO("Config recieved: " << jsonConfigValue);
            updateConfig(*config);
        }
    } else if (configName == DEMO_MODE_SOUND_URL) {
        const auto config = tryParseJson(jsonConfigValue);
        if (config.has_value() && config->isString()) {
            YIO_LOG_INFO("Config recieved: " << jsonConfigValue);
            updateModule(*config);
        }
    }
}

void DemoConfigManager::updateConfig(const Json::Value& value) {
    std::lock_guard<std::mutex> lock(mutex_);

    httpClientTimeoutMs_ = tryGetInt(value, "httpClientTimeoutMs", httpClientTimeoutMs_);
    httpClientRetriesCount_ = tryGetInt(value, "httpClientRetriesCount", httpClientRetriesCount_);
    if (value.isMember("pauseBetweenStoriesMs") && value["pauseBetweenStoriesMs"].isInt()) {
        if (pauseBetweenItemsMs_.has_value()) {
            *pauseBetweenItemsMs_ = std::chrono::milliseconds(getInt(value, "pauseBetweenStoriesMs"));
        } else {
            pauseBetweenItemsMs_ = std::make_optional<std::chrono::milliseconds>(getInt(value, "pauseBetweenStoriesMs"));
        }
    }

    if (updateModule(value)) {
        YIO_LOG_INFO("Demo module updated to: " << jsonToString(module_->toJson()));
        notifyDemoModeReady();
    } else {
        YIO_LOG_ERROR_EVENT("DemoConfigManager.UpdateConfigFailed", "Can't update config to: " << jsonToString(value));
        cleanCacheDirectory();
        notifyDemoModeError();
    }
}

bool DemoConfigManager::checkCachedFiles(const std::shared_ptr<IDemoProvider>& module) {
    std::map<std::string, std::string> filesToDownload = getFilesToDownload(module);
    for (const auto& [url, path] : filesToDownload) {
        YIO_LOG_INFO("Check cache for " << path);
        if (!fileExists(path)) {
            YIO_LOG_INFO("File is missing " << path);
            return false;
        }
    }
    return true;
}

bool DemoConfigManager::updateModule(const Json::Value& value) {
    const auto prevJsonModuleValue = module_->toJson();
    module_->updateDemoItems(value);
    if (module_->toJson() == prevJsonModuleValue && checkCachedFiles(module_)) {
        YIO_LOG_INFO("The same module config received, nothing to update");
        return true;
    }

    if (!downloadDemoFiles(module_)) {
        YIO_LOG_ERROR_EVENT("DemoConfigManager.DownloadDemoFilesFailed", "Can't download demo files for module");
        return false;
    }

    YIO_LOG_INFO("Try to persist module config to: " << configPath_);
    PersistentFile file(configPath_, PersistentFile::Mode::TRUNCATE);
    bool result = file.write(jsonToString(module_->toJson()));
    if (result) {
        YIO_LOG_INFO("Module persisted to " << configPath_);
    } else {
        YIO_LOG_ERROR_EVENT("DemoConfigManager.SaveDemoModuleConfigFailed", "Can't persist module to " << configPath_);
    }
    return result;
}

bool DemoConfigManager::downloadDemoFiles(const std::shared_ptr<IDemoProvider>& module) {
    YIO_LOG_INFO("Try to download demo files");

    cleanCacheDirectory();

    std::vector<HttpClient::DownloadFileTask> tasks;
    std::map<std::string, std::string> filesToDownload = getFilesToDownload(module);

    tasks.reserve(filesToDownload.size());
    for (const auto& [url, path] : filesToDownload) {
        YIO_LOG_INFO("Will download " << url << " into " << path);
        tasks.push_back(quasar::HttpClient::DownloadFileTask{.url = url, .outputFileName = path});
    }

    HttpClient httpClient("mds", device_);
    httpClient.setTimeout(std::chrono::milliseconds{httpClientTimeoutMs_});
    httpClient.setConnectionTimeout(std::chrono::milliseconds(httpClientTimeoutMs_));
    httpClient.setRetriesCount(httpClientRetriesCount_);

    try {
        YIO_LOG_INFO("Start downloading...");
        httpClient.download("demo-mode-story", tasks);
        YIO_LOG_INFO("Download done.");
    } catch (const std::runtime_error& e) {
        YIO_LOG_ERROR_EVENT("DemoConfigManager.DownloadFailed", "Can't download demo files, error: " << e.what());
        return false;
    }

    return true;
}

std::map<std::string, std::string> DemoConfigManager::getFilesToDownload(const std::shared_ptr<IDemoProvider>& module) {
    std::map<std::string, std::string> result;
    for (const auto& item : module->getAllDemoItems()) {
        result[item.sound.url] = item.sound.cachedFile;
        if (item.animation.has_value()) {
            result[item.animation->in.url] = item.animation->in.cachedFile;
            result[item.animation->loop.url] = item.animation->loop.cachedFile;
            result[item.animation->out.url] = item.animation->out.cachedFile;
        }
    }
    return result;
}

void DemoConfigManager::cleanCacheDirectory() {
    if (cachePath_.Exists()) {
        YIO_LOG_INFO("Cleaning cache directory " << cachePath_);
        cachePath_.ForceDelete();
    }

    YIO_LOG_INFO("Creating cache directory " << cachePath_.GetPath());
    cachePath_.MkDirs();
}

std::deque<DemoItem> DemoConfigManager::getNextDemoItems() {
    std::lock_guard<std::mutex> lock(mutex_);
    return module_->getNextDemoItems();
}

std::deque<DemoItem> DemoConfigManager::getAllDemoItems() const {
    std::lock_guard<std::mutex> lock(mutex_);
    return module_->getAllDemoItems();
}

std::optional<std::chrono::milliseconds> DemoConfigManager::getPauseBetweenItems() const {
    std::lock_guard<std::mutex> lock(mutex_);
    return pauseBetweenItemsMs_;
}

// NOLINTNEXTLINE(readability-convert-member-functions-to-static)
void DemoConfigManager::subscribeToSystemConfig(YandexIO::SDKInterface& sdk) {
    sdk.subscribeToSystemConfig(DEMO_MODE_CONFIG);
    sdk.subscribeToSystemConfig(DEMO_MODE_SOUND_URL);
}

void DemoConfigManager::addListener(std::weak_ptr<IDemoModeListener> listener) {
    std::scoped_lock guard(listenerMutex_);
    listeners_.push_back(std::move(listener));
}

std::list<std::weak_ptr<IDemoModeListener>> DemoConfigManager::getListeners() const {
    std::scoped_lock guard(listenerMutex_);
    return listeners_;
}

void DemoConfigManager::notifyDemoModeError() {
    const auto listeners = getListeners();
    for (const auto& wlistener : listeners) {
        if (auto listener = wlistener.lock()) {
            listener->onDemoError();
        }
    }
}

void DemoConfigManager::notifyDemoModeReady() {
    const auto listeners = getListeners();
    for (const auto& wlistener : listeners) {
        if (auto listener = wlistener.lock()) {
            listener->onDemoReady();
        }
    }
}
