#include "demo_mode_handler.h"

#include <yandex_io/capabilities/file_player/play_sound_file_listener.h>

#include <yandex_io/libs/base/utils.h>
#include <yandex_io/libs/json_utils/json_utils.h>
#include <yandex_io/libs/logging/logging.h>

#include <fstream>

DemoModeHandler::DemoModeHandler(std::shared_ptr<IDemoProvider> provider,
                                 std::shared_ptr<YandexIO::SDKInterface> sdk,
                                 std::string demoModePath)
    : provider_(std::move(provider))
    , sdk_(std::move(sdk))
    , blocker_(sdk_, "demo_mode")
    , demoModePath_(std::move(demoModePath))
    , isDemoMode_(quasar::fileExists(demoModePath_))
    , callbackQueue_(std::make_shared<quasar::NamedCallbackQueue>("DemoModeHandler"))
    , playSoundFileListener_(std::make_shared<YandexIO::PlaySoundFileListener>(nullptr, makeSafeCallback([this]() {
                                                                                   onItemFinished();
                                                                               }, lifetime_, callbackQueue_)))
{
    if (isDemoMode()) {
        blocker_.block();
        provider_->prepareItems();
    }
}

DemoModeHandler::~DemoModeHandler() {
    callbackQueue_->add([this]() {
        sdk_->getFilePlayerCapability()->stopSoundFile(playingItem_);
    }, lifetime_);
    callbackQueue_->wait();
    lifetime_.die();
    callbackQueue_->destroy();
}

void DemoModeHandler::playItem() {
    items_ = provider_->getNextDemoItems();
    const std::shared_ptr<YandexIO::IPlaySoundFileListener> playSoundFileListener_;
    scheduleItemsToPlay();
}

void DemoModeHandler::stopItem() {
    const auto& first = items_.front();
    YIO_LOG_INFO("Cancel item: " << first.sound.cachedFile);
    notifyDemoModeItemFinished(first, false);
    sdk_->getFilePlayerCapability()->stopSoundFile(playingItem_);
    items_.clear();
}

void DemoModeHandler::play() {
    callbackQueue_->add([this]() {
        playItem();
    }, lifetime_);
}

void DemoModeHandler::pause() {
    callbackQueue_->add([this]() {
        if (items_.empty()) {
            return;
        }
        stopItem();
    }, lifetime_);
}

void DemoModeHandler::togglePlay() {
    callbackQueue_->add([this]() {
        if (items_.empty()) {
            playItem();
        } else {
            stopItem();
        }
    }, lifetime_);
}

void DemoModeHandler::scheduleItemsToPlay() {
    if (items_.empty()) {
        YIO_LOG_INFO("No items to play");
        return;
    }

    const auto& item = items_.front();
    YIO_LOG_INFO("Item to play: " << item.sound.cachedFile);
    notifyDemoModeItemPlaying(item);

    playingItem_ = item.sound.cachedFile;
    sdk_->getFilePlayerCapability()->playSoundFile(playingItem_,
                                                   quasar::proto::AudioChannel::DIALOG_CHANNEL,
                                                   std::nullopt,
                                                   playSoundFileListener_);
}

void DemoModeHandler::onItemFinished() {
    Y_ENSURE_THREAD(callbackQueue_);
    if (items_.empty()) {
        return;
    }

    const auto& story = items_.front();
    YIO_LOG_INFO("Sound finished playing: " << story.sound.cachedFile);
    notifyDemoModeItemFinished(story, items_.size() > 1);
    items_.pop_front();
    auto pause = provider_->getPauseBetweenItems();
    if (!pause.has_value()) {
        return;
    }
    callbackQueue_->addDelayed([this]() {
        scheduleItemsToPlay();
    }, *pause, lifetime_);
}

void DemoModeHandler::toggleDemoMode() {
    if (isDemoMode()) {
        YIO_LOG_INFO("Deactivating demo mode");
        blocker_.unblock();
        isDemoMode_.store(false);
        sdk_->toggleSetupMode();
        notifyDemoModeEnd();
        std::remove(demoModePath_.c_str());
    } else {
        YIO_LOG_INFO("Activating demo mode");
        blocker_.block();
        isDemoMode_.store(true);
        sdk_->toggleSetupMode();
        notifyDemoModeStart();
        provider_->prepareItems();
        std::ofstream demoModeFile{demoModePath_};
    }
}

bool DemoModeHandler::isDemoMode() const {
    return isDemoMode_.load();
}

void DemoModeHandler::addListener(std::weak_ptr<IDemoModeListener> listener) {
    if (!onListenerAdded(listener)) {
        return;
    }
    std::scoped_lock guard(listenersMutex_);
    listeners_.push_back(std::move(listener));
}

bool DemoModeHandler::onListenerAdded(std::weak_ptr<IDemoModeListener> wlistener) const {
    if (auto listener = wlistener.lock()) {
        if (isDemoMode()) {
            listener->onDemoModeStart();
        } else {
            listener->onDemoModeEnd();
        }
        return true;
    }
    return false;
}

void DemoModeHandler::notifyDemoModeStart() {
    const auto listeners = getListeners();
    for (const auto& wlistener : listeners) {
        if (auto listener = wlistener.lock()) {
            listener->onDemoModeEnd();
        }
    }
}

void DemoModeHandler::notifyDemoModeEnd() {
    const auto listeners = getListeners();
    for (const auto& wlistener : listeners) {
        if (auto listener = wlistener.lock()) {
            listener->onDemoModeEnd();
        }
    }
}

void DemoModeHandler::notifyDemoModeItemPlaying(const DemoItem& item) {
    const auto listeners = getListeners();
    for (const auto& wlistener : listeners) {
        if (auto listener = wlistener.lock()) {
            listener->onDemoModeItemPlaying(item);
        }
    }
}

void DemoModeHandler::notifyDemoModeItemFinished(const DemoItem& item, bool pendingStories) {
    const auto listeners = getListeners();
    for (const auto& wlistener : listeners) {
        if (auto listener = wlistener.lock()) {
            listener->onDemoModeItemFinished(item, pendingStories);
        }
    }
}

std::list<std::weak_ptr<IDemoModeListener>> DemoModeHandler::getListeners() {
    std::scoped_lock guard(listenersMutex_);
    return listeners_;
}

void DemoModeHandler::onCommand(std::string_view command) {
    if (command == "demo_mode") {
        toggleDemoMode();
    }
}
