#include "extended_device_state.h"

namespace YandexIO {

    ExtendedDeviceState::SuggestedLedAnimation computeSuggestedLedAnimation(const ExtendedDeviceState& deviceState) {
        if (deviceState.sdkState.isAlarmPlaying) {
            return ExtendedDeviceState::SuggestedLedAnimation::ALARM;
        }
        if (deviceState.sdkState.isTimerPlaying) {
            return ExtendedDeviceState::SuggestedLedAnimation::TIMER;
        }
        if (deviceState.sdkState.updateState.isCritical) {
            switch (deviceState.sdkState.updateState.state) {
                case YandexIO::SDKState::UpdateState::State::DOWNLOADING:
                    return ExtendedDeviceState::SuggestedLedAnimation::UPDATE_DOWNLOADING;
                case YandexIO::SDKState::UpdateState::State::APPLYING:
                    return ExtendedDeviceState::SuggestedLedAnimation::UPDATE_APPLYING;
                case YandexIO::SDKState::UpdateState::State::NONE:
                    break;
            }
        }
        /* Show data reset event if mics are muted */
        if (deviceState.dataResetState.has_value()) {
            switch (*deviceState.dataResetState) {
                case ExtendedDeviceState::DataResetState::DATA_RESET_WAIT_CONFIRM:
                    return ExtendedDeviceState::SuggestedLedAnimation::DATA_RESET_WAIT_CONFIRM;
                case ExtendedDeviceState::DataResetState::DATA_RESET_IN_PROGRESS:
                    return ExtendedDeviceState::SuggestedLedAnimation::DATA_RESET_IN_PROGRESS;
                default:
                    break;
            }
        }
        if (deviceState.isMicOn.has_value() && *deviceState.isMicOn == false) {
            return ExtendedDeviceState::SuggestedLedAnimation::MUTE;
        }

        switch (deviceState.sdkState.aliceState.state) {
            case YandexIO::SDKState::AliceState::State::SPEAKING:
                return ExtendedDeviceState::SuggestedLedAnimation::SPEAKING;
            case YandexIO::SDKState::AliceState::State::LISTENING:
                return ExtendedDeviceState::SuggestedLedAnimation::LISTENING;
            case YandexIO::SDKState::AliceState::State::THINKING:
                return ExtendedDeviceState::SuggestedLedAnimation::THINKING;
            default:
                break;
        }

        if (deviceState.bluetoothStreamInState.has_value() && deviceState.bluetoothStreamInState.value().isPlaying) {
            return ExtendedDeviceState::SuggestedLedAnimation::BLUETOOTH_IS_PLAYING;
        }

        /* Handle Configuration State */
        switch (deviceState.sdkState.configurationState) {
            case YandexIO::SDKState::ConfigurationState::CONFIGURING:
                return ExtendedDeviceState::SuggestedLedAnimation::CONFIGURING;
            default:
                break;
        }

        if (deviceState.sdkState.notificationState == YandexIO::SDKState::NotificationState::AVAILABLE) {
            return ExtendedDeviceState::SuggestedLedAnimation::NOTIFICATION_AVAILABLE;
        }

        switch (deviceState.sdkState.iotState) {
            case YandexIO::SDKState::IotState::STARTING_DISCOVERY:
            case YandexIO::SDKState::IotState::DISCOVERY_IN_PROGRESS:
                return ExtendedDeviceState::SuggestedLedAnimation::IOT_DISCOVERY_IN_PROGRESS;
            default:
                break;
        }

        switch (deviceState.sdkState.callState.status) {
            case YandexIO::SDKState::CallState::Status::NEW:
            case YandexIO::SDKState::CallState::Status::DIALING:
            case YandexIO::SDKState::CallState::Status::RINGING:
            case YandexIO::SDKState::CallState::Status::ACCEPTING:
                switch (deviceState.sdkState.callState.direction) {
                    case YandexIO::SDKState::CallState::Direction::INCOMING:
                        return ExtendedDeviceState::SuggestedLedAnimation::CALL_RINGING_INCOMING;
                    case YandexIO::SDKState::CallState::Direction::OUTGOING:
                        return ExtendedDeviceState::SuggestedLedAnimation::CALL_RINGING_OUTGOING;
                }
            case YandexIO::SDKState::CallState::Status::CONNECTING:
                return ExtendedDeviceState::SuggestedLedAnimation::CALL_CONNECTING;
            case YandexIO::SDKState::CallState::Status::CONNECTED:
                return ExtendedDeviceState::SuggestedLedAnimation::CALL_CONNECTED;
            case YandexIO::SDKState::CallState::Status::ENDED_OK:
                return ExtendedDeviceState::SuggestedLedAnimation::CALL_ENDED_OK;
            case YandexIO::SDKState::CallState::Status::ENDED_FAILURE:
                return ExtendedDeviceState::SuggestedLedAnimation::CALL_ENDED_FAILURE;
            default:
                break;
        }

        if (!deviceState.sdkState.timers.empty()) {
            return ExtendedDeviceState::SuggestedLedAnimation::TIMER_COUNTDOWN;
        }

        if (deviceState.sdkState.multiroomState.slaveClockSyncing) {
            if (deviceState.sdkState.multiroomState.slaveSyncLevel == YandexIO::SDKState::MultiroomState::SyncLevel::NONE) {
                return ExtendedDeviceState::SuggestedLedAnimation::MULTIROOM_CLOCK_IS_SYNCHRONIZING1;
            } else if (deviceState.sdkState.multiroomState.slaveSyncLevel == YandexIO::SDKState::MultiroomState::SyncLevel::WEAK) {
                return ExtendedDeviceState::SuggestedLedAnimation::MULTIROOM_CLOCK_IS_SYNCHRONIZING2;
            }
        }

        if (deviceState.sdkState.playerState.isAudioPlaying() || deviceState.sdkState.multiroomState.playing) {
            return ExtendedDeviceState::SuggestedLedAnimation::AUDIO_PLAYING;
        }

        if (deviceState.bluetoothStreamInState.has_value() && deviceState.bluetoothStreamInState->isDiscoverable) {
            return ExtendedDeviceState::SuggestedLedAnimation::BLUETOOTH_DISCOVERY;
        }

        if (deviceState.sdkState.notificationState == YandexIO::SDKState::NotificationState::PASSIVE) {
            return ExtendedDeviceState::SuggestedLedAnimation::NOTIFICATION_PASSIVE;
        }

        return ExtendedDeviceState::SuggestedLedAnimation::IDLE;
    }

    bool ExtendedDeviceState::isMediaPlaying() const {
        return sdkState.isMediaPlaying() || bluetoothStreamInState.value_or(BluetoothStreamInState{}).isPlaying;
    }

    bool ExtendedDeviceState::isBluetoothPlaying() const {
        return bluetoothStreamInState.value_or(BluetoothStreamInState{}).isPlaying;
    }

    std::string ExtendedDeviceState::aliceStateToString(YandexIO::SDKState::AliceState aliceState) {
        return YandexIO::SDKState::aliceStateToString(aliceState);
    }

    std::string ExtendedDeviceState::configurationStateToString(YandexIO::SDKState::ConfigurationState configurationState) {
        return YandexIO::SDKState::configurationStateToString(configurationState);
    }

    std::string ExtendedDeviceState::updateStateToString(YandexIO::SDKState::UpdateState updateState) {
        return YandexIO::SDKState::updateStateToString(updateState);
    }

    std::string ExtendedDeviceState::dataResetStateToString(ExtendedDeviceState::DataResetState dataResetState) {
        switch (dataResetState) {
            case DataResetState::DATA_RESET_IN_PROGRESS:
                return "Data reset in progress";
            case DataResetState::DATA_RESET_WAIT_CONFIRM:
                return "Data reset wait confirm";
            case DataResetState::NONE:
                return "Data reset is None";
        }
    }

} // namespace YandexIO
