#include "equalizer_controller.h"

#include <yandex_io/libs/base/directives.h>
#include <yandex_io/libs/json_utils/json_utils.h>
#include <yandex_io/libs/logging/logging.h>
#include <yandex_io/modules/equalizer_controller/dispatcher/dummy/dummy_equalizer_dispatcher.h>

#include <alice/protos/endpoint/capability.pb.h>

#include <future>

YIO_DEFINE_LOG_MODULE("equalizer_controller");

namespace YandexIO {

    namespace {

        constexpr auto CONFIG_NAME = "equalizer";
        constexpr int MIN_BANDS_COUNT = 1;
        constexpr int MAX_BANDS_COUNT = 64;
        constexpr double MAX_ABS_GAIN = 6.;
        constexpr auto ROOM_CORRECTION_CONFIG_NAME = "roomCorrection";

        EqualizerConfig adjustableDirectiveToEqualizer(const Json::Value& payload) {
            EqualizerConfig result;
            const auto& bandsJson = payload["bands"];
            for (size_t i = 0; i < bandsJson.size(); ++i) {
                const auto& bandJson = bandsJson.get(i, Json::Value());
                result.bands.push_back(EqualizerConfig::Band{
                    .freq = bandJson["frequency"].asDouble(),
                    .width = bandJson["width"].asDouble(),
                    .gain = bandJson["gain"].asDouble()});
            }
            return result;
        }

        EqualizerConfig fixedDirectiveToEqualizer(const Json::Value& payload, const EqualizerDispatcher::BandsConfiguration& bandConfig) {
            EqualizerConfig result;
            const auto& gainsJson = payload["gains"];
            if (gainsJson.size() != bandConfig.size()) {
                YIO_LOG_WARN("Different sizes of received gains and fixed bands configuration");
                return result;
            }

            for (size_t i = 0; i < gainsJson.size(); ++i) {
                result.bands.push_back(EqualizerConfig::Band{
                    .freq = bandConfig[i].freq,
                    .width = bandConfig[i].width,
                    .gain = gainsJson[static_cast<int>(i)].asDouble()});
            }
            return result;
        }

    } // namespace

    const std::string EqualizerController::FIXED_BANDS_DIRECTIVE = "set_fixed_equalizer_bands_directive";
    const std::string EqualizerController::ADJUSTABLE_BANDS_DIRECTIVE = "set_adjustable_equalizer_bands_directive";

    EqualizerController::EqualizerController(std::unique_ptr<quasar::ICallbackQueue> callbackQueue,
                                             const Json::Value& audioConfig,
                                             std::weak_ptr<SDKInterface> sdk,
                                             std::weak_ptr<YandexIO::ITelemetry> telemetry,
                                             std::unique_ptr<IEqualizerDispatcherFactory> dispatcherFactory)
        : callbackQueue_(std::move(callbackQueue))
        , sdk_(std::move(sdk))
        , telemetry_(std::move(telemetry))
        , dispatcherFactory_(std::move(dispatcherFactory))
        , dispatcher_(std::make_unique<DummyEqualizerDispatcher>())
        , smartEqualizer_(quasar::tryGetJson(audioConfig, ROOM_CORRECTION_CONFIG_NAME))
        , capabilityState_(createCapabilityState())
    {
    }

    EqualizerController::~EqualizerController() {
        callbackQueue_.reset();
    }

    void EqualizerController::subscribeToBackendConfig() {
        if (auto sdk = sdk_.lock()) {
            sdk->subscribeToSystemConfig(CONFIG_NAME);
            sdk->subscribeToDeviceConfig(CONFIG_NAME);
        }
    }

    void EqualizerController::onSystemConfig(const std::string& configName, const std::string& jsonConfigValue) {
        if (configName != CONFIG_NAME) {
            return;
        }

        callbackQueue_->add([this, jsonConfigValue] {
            Json::Value jsonConfig;
            try {
                jsonConfig = quasar::parseJson(jsonConfigValue);
            } catch (const Json::Exception& e) {
                YIO_LOG_ERROR_EVENT("EqualizerController.InvalidSystemConfigJson", e.what());
                return;
            }

            auto newSystemConfig = SystemConfig::fromJson(jsonConfig);
            if (systemConfig_ == newSystemConfig) {
                return;
            }

            YIO_LOG_INFO("New equalizer system config: " << jsonConfigValue);
            std::swap(systemConfig_, newSystemConfig);

            if (systemConfig_.roomCorrectionConfig != newSystemConfig.roomCorrectionConfig) {
                smartEqualizer_.setConfig(systemConfig_.roomCorrectionConfig);
            }

            if (systemConfig_.type != newSystemConfig.type) {
                updateDispatcher();
            }

            // defaults for RC & MC may have changed, so reload device config
            processDeviceConfig();
        });
    }

    void EqualizerController::onDeviceConfig(const std::string& configName, const std::string& jsonConfigValue) {
        if (configName != CONFIG_NAME) {
            return;
        }

        callbackQueue_->add([this, jsonConfigValue] {
            Json::Value jsonConfig;
            try {
                jsonConfig = quasar::parseJson(jsonConfigValue);
            } catch (const Json::Exception& e) {
                YIO_LOG_ERROR_EVENT("EqualizerController.InvalidDeviceConfigJson", e.what());
                return;
            }

            if (deviceConfig_ == jsonConfig) {
                return;
            }
            YIO_LOG_INFO("New equalizer device config: " << jsonConfigValue);
            deviceConfig_ = std::move(jsonConfig);
            processDeviceConfig();
        });
    }

    void EqualizerController::onAudioData(const ChannelsData& data) {
        callbackQueue_->add([this, data] {
            smartEqualizer_.processAudioData(data);

            if (!roomCorrectionEnabled() || audioPlaying_) {
                return;
            }

            pullSmartConfig();
        });
    }

    void EqualizerController::onSDKState(const SDKState& state) {
        callbackQueue_->add([this, state = state.playerState] {
            const bool newAudioPlaying = state.isAudioPlaying();
            if (newAudioPlaying == audioPlaying_) {
                return;
            }

            audioPlaying_ = newAudioPlaying;

            if (!audioPlaying_ && roomCorrectionEnabled()) {
                // pull & apply smart equalizer in pause between tracks if it enabled
                pullSmartConfig();
            }
        });
    }

    NAlice::TCapabilityHolder EqualizerController::getState() const {
        if (callbackQueue_->isWorkingThread()) {
            return capabilityState_;
        }

        std::promise<NAlice::TCapabilityHolder> promise;
        callbackQueue_->add([this, &promise] {
            promise.set_value(capabilityState_);
        });
        return promise.get_future().get();
    }

    IDirectiveHandlerPtr EqualizerController::getDirectiveHandler() {
        return shared_from_this();
    }

    void EqualizerController::addListener(std::weak_ptr<IListener> wlistener) {
        callbackQueue_->add([this, wlistener{std::move(wlistener)}] {
            capabilityListeners_.insert(wlistener);
        });
    }
    void EqualizerController::removeListener(std::weak_ptr<IListener> wlistener) {
        callbackQueue_->add([this, wlistener{std::move(wlistener)}] {
            capabilityListeners_.erase(wlistener);
        });
    }

    const std::string& EqualizerController::getHandlerName() const {
        static const std::string CAPABILITY_NAME = "EqualizerCapability";
        return CAPABILITY_NAME;
    }

    const std::set<std::string>& EqualizerController::getSupportedDirectiveNames() const {
        static const std::set<std::string> DIRECTIVE_NAMES = {FIXED_BANDS_DIRECTIVE, ADJUSTABLE_BANDS_DIRECTIVE};
        return DIRECTIVE_NAMES;
    }

    void EqualizerController::handleDirective(const std::shared_ptr<Directive>& directive) {
        callbackQueue_->add([this, directive] {
            if (userConfig_.presetMode != UserConfig::PresetMode::MEDIA_CORRECTION) {
                YIO_LOG_WARN("Media correction disabled. Skipping directive");
                return;
            }

            if (directive->is(FIXED_BANDS_DIRECTIVE)) {
                handleFixedDirective(directive->getData());
            } else if (directive->is(ADJUSTABLE_BANDS_DIRECTIVE)) {
                handleAdjustableDirective(directive->getData());
            } else {
                YIO_LOG_WARN("Unsupported directive");
                return;
            }
        });
    }

    void EqualizerController::cancelDirective(const std::shared_ptr<Directive>& /*directive*/) {
        // ¯\_(ツ)_/¯
    }
    void EqualizerController::prefetchDirective(const std::shared_ptr<Directive>& /*directive*/) {
        // ¯\_(ツ)_/¯
    }

    void EqualizerController::onDirective(
        const std::string& name, const std::string& /*vinsRequestId*/, const std::string& /*jsonPayload*/)
    {
        static const std::set<std::string> DIRECTIVE_NAMES = {
            quasar::Directives::AUDIO_STOP,
            quasar::Directives::CLEAR_QUEUE};

        if (DIRECTIVE_NAMES.contains(name)) {
            // reset media correction after pause
            mediaCorrectionConfig_ = EqualizerConfig{};

            if (userConfig_.presetMode == UserConfig::PresetMode::MEDIA_CORRECTION) {
                YIO_LOG_INFO("Provide empty media correction config");
                provideEqualizer(mediaCorrectionConfig_);
            }
        }
    }

    bool EqualizerController::roomCorrectionEnabled() const {
        return smartEqualizer_.initialized() && userConfig_.roomCorrection;
    }

    void EqualizerController::processDeviceConfig() {
        UserConfig newUserConfig = UserConfig::fromJson(deviceConfig_, systemConfig_.roomCorrectionDefault,
                                                        systemConfig_.mediaCorrectionDefault);
        if (newUserConfig == userConfig_) {
            return;
        }

        YIO_LOG_INFO("New equalizer user config: " << newUserConfig);
        userConfig_ = std::move(newUserConfig);

        updateState();

        switch (userConfig_.presetMode) {
            case UserConfig::PresetMode::DEFAULT:
                provideEqualizer(EqualizerConfig{});
                break;
            case UserConfig::PresetMode::USER:
                provideEqualizer(userConfig_.equalizerConfig);
                break;
            case UserConfig::PresetMode::MEDIA_CORRECTION:
                if (mediaCorrectionConfig_ == EqualizerConfig{} && audioPlaying_) {
                    requestMediaCorrectionConfig();
                }
                provideEqualizer(mediaCorrectionConfig_);
                break;
        }
    }

    void EqualizerController::updateDispatcher() {
        // set default settings to old equalizer
        dispatcher_->setUserConfig(EqualizerConfig{});

        try {
            dispatcher_ = dispatcherFactory_->createDispatcher(systemConfig_.type);
        } catch (const std::runtime_error& err) {
            YIO_LOG_ERROR_EVENT("EqualizerController.CreateDispatcher.Exception", err.what());
            dispatcher_ = std::make_unique<DummyEqualizerDispatcher>();
        }

        // set cached config to new equalizer
        if (roomCorrectionEnabled()) {
            dispatcher_->setSmartConfig(smartEqualizerConfig_);
        } else if (userConfig_.presetMode == UserConfig::PresetMode::USER) {
            dispatcher_->setUserConfig(userConfig_.equalizerConfig);
        } else if (userConfig_.presetMode == UserConfig::PresetMode::MEDIA_CORRECTION) {
            dispatcher_->setUserConfig(mediaCorrectionConfig_);
        }

        updateState();
    }

    void EqualizerController::provideEqualizer(const EqualizerConfig& config) {
        if (roomCorrectionEnabled()) {
            smartEqualizer_.setUserEqualizerConfig(config);
            pullSmartConfig();
        } else {
            dispatcher_->setUserConfig(config);
        }
    }

    void EqualizerController::pullSmartConfig() {
        smartEqualizerState_ = smartEqualizer_.toString();
        if (auto config = smartEqualizer_.getConfig()) {
            smartEqualizerConfig_ = config.value();
            YIO_LOG_INFO("Received new config from smart equalizer: " << smartEqualizerConfig_);
            dispatcher_->setSmartConfig(smartEqualizerConfig_);
            stats_.onSmartConfig(smartEqualizerConfig_, smartEqualizer_.getMicAndFeedbackRms());
            tryToSendStats();
        }
    }

    void EqualizerController::tryToSendStats() {
        if (stats_.isEnoughDataToSend()) {
            YIO_LOG_INFO("Enough equalizer data has been accumulated. Reporting it to telemetry");
            auto eventValue = stats_.formatJson();
            eventValue["smartEqualizerState"] = quasar::tryParseJson(smartEqualizerState_).value_or(smartEqualizerState_);
            if (auto telemetry = telemetry_.lock()) {
                telemetry->reportEvent("equalizerStats", quasar::jsonToString(eventValue));
            }
            stats_.reset();
        }
    }

    NAlice::TCapabilityHolder EqualizerController::createCapabilityState() const {
        NAlice::TCapabilityHolder capabilityHolder;
        auto equalizerCapability = capabilityHolder.mutable_equalizercapability();

        const auto bandsConfig = dispatcher_->getFixedBandsConfiguration();

        auto meta = equalizerCapability->mutable_meta();
        if (bandsConfig.empty()) {
            meta->add_supporteddirectives(NAlice::TCapability::SetAdjustableEqualizerBandsDirectiveType);
        } else {
            meta->add_supporteddirectives(NAlice::TCapability::SetFixedEqualizerBandsDirectiveType);
        }

        auto parameters = equalizerCapability->mutable_parameters();
        auto bandsLimit = parameters->mutable_bandslimits();
        bandsLimit->set_minbandscount(bandsConfig.empty() ? MIN_BANDS_COUNT : bandsConfig.size());
        bandsLimit->set_maxbandscount(bandsConfig.empty() ? MAX_BANDS_COUNT : bandsConfig.size());
        bandsLimit->set_minbandgain(-MAX_ABS_GAIN);
        bandsLimit->set_maxbandgain(MAX_ABS_GAIN);
        if (bandsConfig.empty()) {
            parameters->mutable_adjustable();
        } else {
            auto fixed = parameters->mutable_fixed();
            for (auto& bandConfig : bandsConfig) {
                auto fixedBand = fixed->add_fixedbands();
                fixedBand->set_frequency(bandConfig.freq);
                fixedBand->set_width(bandConfig.width);
            }
        }

        auto state = equalizerCapability->mutable_state();
        switch (userConfig_.presetMode) {
            case UserConfig::PresetMode::DEFAULT:
                state->set_presetmode(NAlice::TEqualizerCapability::Default);
                break;
            case UserConfig::PresetMode::USER:
                state->set_presetmode(NAlice::TEqualizerCapability::User);
                for (const auto& band : userConfig_.equalizerConfig.bands) {
                    auto bandProto = state->add_bands();
                    bandProto->set_frequency(band.freq);
                    bandProto->set_gain(band.gain);
                    bandProto->set_width(band.width);
                }
                break;
            case UserConfig::PresetMode::MEDIA_CORRECTION:
                state->set_presetmode(NAlice::TEqualizerCapability::MediaCorrection);
                for (const auto& band : mediaCorrectionConfig_.bands) {
                    auto bandProto = state->add_bands();
                    bandProto->set_frequency(band.freq);
                    bandProto->set_gain(band.gain);
                    bandProto->set_width(band.width);
                }
                break;
        }

        return capabilityHolder;
    }

    void EqualizerController::updateState() {
        capabilityState_ = createCapabilityState();

        for (const auto& wlistener : capabilityListeners_) {
            if (auto listener = wlistener.lock()) {
                listener->onCapabilityStateChanged(shared_from_this(), capabilityState_);
            }
        }
    }

    void EqualizerController::handleFixedDirective(const Directive::Data& directiveData) {
        auto bandsConfig = dispatcher_->getFixedBandsConfiguration();
        if (bandsConfig.empty()) {
            YIO_LOG_WARN("There is no fixed bands configuration and fixed directive was received");
            return;
        }

        mediaCorrectionConfig_ = fixedDirectiveToEqualizer(directiveData.payload, bandsConfig);
        YIO_LOG_INFO("Fixed bands directive received. " << mediaCorrectionConfig_);

        provideEqualizer(mediaCorrectionConfig_);
        updateState();
    }

    void EqualizerController::handleAdjustableDirective(const Directive::Data& directiveData) {
        auto bandsConfig = dispatcher_->getFixedBandsConfiguration();
        if (!bandsConfig.empty()) {
            YIO_LOG_WARN("There is fixed bands configuration and adjustable directive was received");
            return;
        }

        mediaCorrectionConfig_ = adjustableDirectiveToEqualizer(directiveData.payload);
        YIO_LOG_INFO("Adjustable bands directive received. " << mediaCorrectionConfig_);

        provideEqualizer(mediaCorrectionConfig_);
        updateState();
    }

    void EqualizerController::requestMediaCorrectionConfig() {
        if (auto sdk = sdk_.lock()) {
            auto request = VinsRequest::createEventRequest(
                VinsRequest::buildEqualizerSettingsRequest(),
                VinsRequest::createSoftwareDirectiveEventSource());
            sdk->getAliceCapability()->startRequest(std::move(request), nullptr);
        }
    }

    EqualizerController::UserConfig EqualizerController::UserConfig::fromJson(const Json::Value& json,
                                                                              bool roomCorrectionDefault, bool mediaCorrectionDefault)
    {
        EqualizerController::UserConfig config;

        if (quasar::tryGetBool(json, "enabled", false)) {
            config.presetMode = EqualizerController::UserConfig::PresetMode::USER;
            config.equalizerConfig = EqualizerConfig::fromJson(json);
        } else if (quasar::tryGetBool(json, "mediaCorrectionEnabled", mediaCorrectionDefault)) {
            config.presetMode = EqualizerController::UserConfig::PresetMode::MEDIA_CORRECTION;
        } else {
            config.presetMode = EqualizerController::UserConfig::PresetMode::DEFAULT;
        }

        config.roomCorrection = quasar::tryGetBool(json, "smartEnabled", roomCorrectionDefault);

        return config;
    }

    std::ostream& operator<<(std::ostream& out, const EqualizerController::UserConfig& userConfig) {
        out << "presetMode=";
        switch (userConfig.presetMode) {
            case EqualizerController::UserConfig::PresetMode::DEFAULT:
                out << "DEFAULT";
                break;
            case EqualizerController::UserConfig::PresetMode::USER:
                out << "USER";
                break;
            case EqualizerController::UserConfig::PresetMode::MEDIA_CORRECTION:
                out << "MEDIA_CORRECTION";
                break;
        }

        out << " roomCorrection=" << userConfig.roomCorrection;

        out << " " << userConfig.equalizerConfig;

        return out;
    }

    EqualizerController::SystemConfig EqualizerController::SystemConfig::fromJson(const Json::Value& json) {
        return {
            .type = quasar::tryGetString(json, "type", ""),
            .roomCorrectionDefault = quasar::tryGetBool(json, "roomCorrectionDefault", false),
            .mediaCorrectionDefault = quasar::tryGetBool(json, "mediaCorrectionDefault", false),
            .roomCorrectionConfig = quasar::tryGetJson(json, ROOM_CORRECTION_CONFIG_NAME)};
    }

} // namespace YandexIO
