#pragma once

#include <yandex_io/libs/equalizer_config/equalizer_config.h>
#include <yandex_io/libs/telemetry/telemetry.h>
#include <yandex_io/libs/threading/i_callback_queue.h>
#include <yandex_io/modules/audio_input/smart_equalizer/smart_equalizer.h>
#include <yandex_io/modules/equalizer_controller/dispatcher/equalizer_dispatcher.h>
#include <yandex_io/modules/equalizer_controller/dispatcher/factory/equalizer_dispatcher_factory.h>
#include <yandex_io/modules/equalizer_controller/equalizer_stats.h>
#include <yandex_io/sdk/audio_source/i_audio_source_client.h>
#include <yandex_io/sdk/backend_config_observer.h>
#include <yandex_io/sdk/directive_observer.h>
#include <yandex_io/sdk/interfaces/i_capability.h>
#include <yandex_io/sdk/sdk_interface.h>
#include <yandex_io/sdk/sdk_state_observer.h>

#include <json/json.h>

#include <memory>
#include <mutex>
#include <ostream>

namespace YandexIO {

    class EqualizerController
        : public BackendConfigObserver,
          public IAudioSourceClient::Listener,
          public SDKStateObserver,
          public ICapability,
          public IDirectiveHandler,
          public YandexIO::DirectiveObserver,
          public std::enable_shared_from_this<EqualizerController> {
    public:
        static const std::string FIXED_BANDS_DIRECTIVE;
        static const std::string ADJUSTABLE_BANDS_DIRECTIVE;

        EqualizerController(std::unique_ptr<quasar::ICallbackQueue> callbackQueue,
                            const Json::Value& audioConfig,
                            std::weak_ptr<SDKInterface> sdk,
                            std::weak_ptr<YandexIO::ITelemetry> telemetry,
                            std::unique_ptr<IEqualizerDispatcherFactory> dispatcherFactory);
        ~EqualizerController();

        void subscribeToBackendConfig();

    public:
        // from BackendConfigObserver
        void onSystemConfig(const std::string& configName, const std::string& jsonConfigValue) override;
        void onDeviceConfig(const std::string& configName, const std::string& jsonConfigValue) override;

        // from IAudioSourceClient::Listener
        void onAudioData(const ChannelsData& data) override;

        // from SDKStateObserver
        void onSDKState(const SDKState& state) override;

        // from ICapability
        NAlice::TCapabilityHolder getState() const override;
        IDirectiveHandlerPtr getDirectiveHandler() override;
        void addListener(std::weak_ptr<IListener> wlistener) override;
        void removeListener(std::weak_ptr<IListener> wlistener) override;

        // from YandexIO::IDirectiveHandler
        const std::string& getHandlerName() const override;
        const std::set<std::string>& getSupportedDirectiveNames() const override;
        void handleDirective(const std::shared_ptr<Directive>& directive) override;
        void cancelDirective(const std::shared_ptr<Directive>& directive) override;
        void prefetchDirective(const std::shared_ptr<Directive>& directive) override;

        // from YandexIO::DirectiveObserver
        void onDirective(const std::string& name, const std::string& vinsRequestId, const std::string& jsonPayload) override;

    private:
        bool roomCorrectionEnabled() const;
        void processDeviceConfig();
        void updateDispatcher();
        void provideEqualizer(const EqualizerConfig& config);
        void pullSmartConfig();
        void tryToSendStats();
        NAlice::TCapabilityHolder createCapabilityState() const;
        void updateState();
        void handleFixedDirective(const Directive::Data& directiveData);
        void handleAdjustableDirective(const Directive::Data& directiveData);
        void requestMediaCorrectionConfig();

    private:
        std::unique_ptr<quasar::ICallbackQueue> callbackQueue_;
        const std::weak_ptr<SDKInterface> sdk_;
        const std::weak_ptr<YandexIO::ITelemetry> telemetry_;
        const std::unique_ptr<IEqualizerDispatcherFactory> dispatcherFactory_;
        std::unique_ptr<EqualizerDispatcher> dispatcher_;
        SmartEqualizer smartEqualizer_;
        EqualizerStats stats_;

        struct UserConfig {
            enum class PresetMode {
                DEFAULT,
                USER,
                MEDIA_CORRECTION
            };
            PresetMode presetMode = PresetMode::DEFAULT;

            EqualizerConfig equalizerConfig;
            bool roomCorrection = false;

            auto operator<=>(const UserConfig&) const = default;
            static UserConfig fromJson(const Json::Value& json, bool roomCorrectionDefault, bool mediaCorrectionDefault);
        };
        friend std::ostream& operator<<(std::ostream& out, const UserConfig& userConfig);
        UserConfig userConfig_;
        Json::Value deviceConfig_;

        struct SystemConfig {
            std::string type;
            bool roomCorrectionDefault = false;
            bool mediaCorrectionDefault = false;
            Json::Value roomCorrectionConfig;

            auto operator<=>(const SystemConfig&) const = default;
            static SystemConfig fromJson(const Json::Value& json);
        };
        SystemConfig systemConfig_;

        EqualizerConfig smartEqualizerConfig_;
        std::string smartEqualizerState_;
        EqualizerConfig mediaCorrectionConfig_;
        bool audioPlaying_ = false;

        using WeakCapabilityListener = std::weak_ptr<YandexIO::ICapability::IListener>;
        std::set<WeakCapabilityListener, std::owner_less<WeakCapabilityListener>> capabilityListeners_;
        NAlice::TCapabilityHolder capabilityState_;
    };

} // namespace YandexIO
