#include "geolocation_backup.h"

#include <yandex_io/libs/base/persistent_file.h>
#include <yandex_io/libs/base/utils.h>
#include <yandex_io/libs/logging/logging.h>
#include <yandex_io/protos/model_objects.pb.h>
#include <util/string/cast.h>

#include <fstream>

YIO_DEFINE_LOG_MODULE("geolocation");

using namespace quasar;
using namespace YandexIO;

GeolocationBackup::GeolocationBackup(
    const TFsPath& backupLocationPath,
    const TFsPath& backupTimezonePath)
    : backupLocationPath_(backupLocationPath)
    , backupTimezonePath_(backupTimezonePath)
{
    Y_VERIFY(backupLocationPath_.IsDefined());
    Y_VERIFY(backupTimezonePath_.IsDefined());

    backupLocationPath_.Parent().MkDirs();
    backupTimezonePath_.Parent().MkDirs();
}

std::optional<Location> GeolocationBackup::loadLocation() const {
    if (!backupLocationPath_.Exists()) {
        return std::nullopt;
    }

    try {
        proto::Location locationProto;
        const auto serializedLocation = getFileContent(backupLocationPath_.GetPath());
        Y_ENSURE(locationProto.ParseFromString(TString(serializedLocation)));
        return Location::fromProto(locationProto);
    } catch (const std::exception& e) {
        YIO_LOG_ERROR_EVENT("GeolocationBackup.FailedReadFromBackupFile.Location", e.what());
    }
    return std::nullopt;
}

std::optional<Timezone> GeolocationBackup::loadTimezone() const {
    if (!backupTimezonePath_.Exists()) {
        return std::nullopt;
    }

    try {
        proto::Timezone timezoneProto;
        const auto serializedTimezone = getFileContent(backupTimezonePath_.GetPath());
        Y_ENSURE(timezoneProto.ParseFromString(TString(serializedTimezone)));
        return Timezone::fromProto(timezoneProto);
    } catch (const std::exception& e) {
        YIO_LOG_ERROR_EVENT("GeolocationBackup.FailedReadFromBackupFile.Timezone", e.what());
    }
    return std::nullopt;
}

void GeolocationBackup::saveLocation(const Location& location) const {
    try {
        TransactionFile file(backupLocationPath_.GetPath());
        file.write(location.toProto().SerializeAsString());
        file.commit();
    } catch (const std::exception& e) {
        YIO_LOG_ERROR_EVENT("GeolocationBackup.FailedWriteToBackupFile.Location", e.what());
    }
}

void GeolocationBackup::saveTimezone(const Timezone& timezone) const {
    try {
        TransactionFile file(backupTimezonePath_.GetPath());
        file.write(timezone.toProto().SerializeAsString());
        file.commit();
    } catch (const std::exception& e) {
        YIO_LOG_ERROR_EVENT("GeolocationBackup.FailedWriteToBackupFile.Timezone", e.what());
    }
}
