#include "geolocation.h"

#include <yandex_io/libs/logging/logging.h>

YIO_DEFINE_LOG_MODULE("geolocation");

using namespace quasar;
using namespace YandexIO;

std::shared_ptr<Geolocation> Geolocation::create(
    std::shared_ptr<SDKInterface> sdk,
    std::shared_ptr<ICallbackQueue> callbackQueue,
    std::shared_ptr<IHttpClient> httpClient,
    const Settings& settings) {
    auto geolocation = std::make_shared<Geolocation>(
        std::move(callbackQueue),
        std::move(httpClient),
        settings);

    sdk->addSDKStateObserver(geolocation);
    sdk->addBackendConfigObserver(geolocation);
    GeolocationConfigProvider::subscribeToConfigUpdate(sdk);

    return geolocation;
}

Geolocation::Geolocation(
    std::shared_ptr<ICallbackQueue> callbackQueue,
    std::shared_ptr<IHttpClient> httpClient,
    const Settings& settings)
    : callbackQueue_(callbackQueue)
    , uniqueCallback_(callbackQueue, UniqueCallback::ReplaceType::INSERT_BACK)
    , backup_(settings.backupLocationPath, settings.backupTimezonePath)
    , locationProvider_(httpClient, settings.lbsBackendUrl, settings.deviceType, settings.deviceId)
    , timezoneProvider_(httpClient, settings.quasarBackendUrl)
    , baseScheduleTime_(settings.baseScheduleTime)
    , errorScheduleTime_(settings.errorScheduleTime)
{
    httpClient->setTimeout(std::chrono::seconds{30});
}

Geolocation::~Geolocation() {
    lifetime_.die();
}

void Geolocation::onSDKState(const SDKState& state) {
    if (!state.wifiList.empty()) {
        callbackQueue_->add([this, state]() {
            wifiList_ = state.wifiList;
            if (!uniqueCallback_.isScheduled()) {
                const auto updateStatus = update();
                schedulePeriodicUpdate(updateStatus);
            }
        }, lifetime_);
    }
}

void Geolocation::onDeviceConfig(const std::string& configName, const std::string& jsonConfigValue) {
    callbackQueue_->add([this, configName, jsonConfigValue]() {
        const auto state = configProvider_.onDeviceConfig(configName, jsonConfigValue);
        if (state == GeolocationConfigProvider::State::HAS_UPDATE) {
            YIO_LOG_INFO("Got new device config: " << jsonConfigValue);
            const auto updateStatus = update();
            schedulePeriodicUpdate(updateStatus);
        }
    }, lifetime_);
}

void Geolocation::onSystemConfig(const std::string& configName, const std::string& jsonConfigValue) {
    callbackQueue_->add([this, configName, jsonConfigValue]() {
        const auto state = configProvider_.onSystemConfig(configName, jsonConfigValue);
        if (state == GeolocationConfigProvider::State::HAS_UPDATE) {
            YIO_LOG_INFO("Got new system config: " << jsonConfigValue);
            const auto updateStatus = update();
            schedulePeriodicUpdate(updateStatus);
        }
    }, lifetime_);
}

void Geolocation::addListener(std::weak_ptr<IListener> listener) {
    callbackQueue_->add([this, wlistener = std::move(listener)]() mutable {
        if (const auto location = backup_.loadLocation(); location.has_value()) {
            notifyLocationChanged(wlistener, *location);
        }
        if (const auto timezone = backup_.loadTimezone(); timezone.has_value()) {
            notifyTimezoneChanged(wlistener, *timezone);
        }
        listeners_.push_back(std::move(wlistener));
    }, lifetime_);
}

Geolocation::UpdateStatus Geolocation::update() {
    Y_ENSURE_THREAD(callbackQueue_);

    auto location = configProvider_.getLocation();
    if (!location.has_value()) {
        location = locationProvider_.request(wifiList_);
    }
    if (!location.has_value()) {
        return UpdateStatus::FAILED;
    }

    auto timezone = configProvider_.getTimezone();
    if (!timezone.has_value()) {
        timezone = timezoneProvider_.request(location.value());
    }
    if (!timezone.has_value()) {
        return UpdateStatus::FAILED;
    }

    backup_.saveLocation(*location);
    backup_.saveTimezone(*timezone);
    notifyAllListeners(*location, *timezone);
    return UpdateStatus::SUCCESS;
}

void Geolocation::schedulePeriodicUpdate(const UpdateStatus& status) {
    Y_ENSURE_THREAD(callbackQueue_);

    const auto timeout = status == UpdateStatus::SUCCESS ? baseScheduleTime_ : errorScheduleTime_;
    uniqueCallback_.executeDelayed([this]() {
        const auto updateStatus = update();
        schedulePeriodicUpdate(updateStatus);
    }, timeout, lifetime_);
}

void Geolocation::notifyLocationChanged(const std::weak_ptr<IListener>& wlistener, const Location& location) const {
    Y_ENSURE_THREAD(callbackQueue_);

    if (const auto listener = wlistener.lock()) {
        try {
            listener->onLocationChanged(location);
        } catch (const std::exception& e) {
            YIO_LOG_ERROR_EVENT("Geolocation.FailedNotifyListener.Location", e.what());
        }
    }
}

void Geolocation::notifyTimezoneChanged(const std::weak_ptr<IListener>& wlistener, const Timezone& timezone) const {
    Y_ENSURE_THREAD(callbackQueue_);

    if (const auto listener = wlistener.lock()) {
        try {
            listener->onTimezoneChanged(timezone);
        } catch (const std::exception& e) {
            YIO_LOG_ERROR_EVENT("Geolocation.FailedNotifyListener.Timezone", e.what());
        }
    }
}

void Geolocation::notifyAllListeners(const Location& location, const Timezone& timezone) const {
    Y_ENSURE_THREAD(callbackQueue_);
    YIO_LOG_INFO("Notifying listeners: location=" << location << ", timezone=" << timezone);

    for (const auto& wlistener : listeners_) {
        notifyLocationChanged(wlistener, location);
        notifyTimezoneChanged(wlistener, timezone);
    }
}
