#include "geolocation_config_provider.h"

#include <yandex_io/libs/json_utils/json_utils.h>
#include <yandex_io/libs/logging/logging.h>

YIO_DEFINE_LOG_MODULE("geolocation");

using namespace quasar;
using namespace YandexIO;

namespace {

    constexpr auto DEVICE_CONFIG_NAME = "location";
    constexpr auto SYSTEM_CONFIG_NAME = "locationdSettings";

} // anonymous namespace

void GeolocationConfigProvider::subscribeToConfigUpdate(std::shared_ptr<SDKInterface> sdk) {
    sdk->subscribeToDeviceConfig(DEVICE_CONFIG_NAME);
    sdk->subscribeToSystemConfig(SYSTEM_CONFIG_NAME);
}

GeolocationConfigProvider::State GeolocationConfigProvider::onDeviceConfig(const std::string& configName, const std::string& jsonConfigValue) {
    if (configName != DEVICE_CONFIG_NAME) {
        return State::NO_UPDATE;
    }

    Json::Value jsonLocation;
    try {
        jsonLocation = parseJson(jsonConfigValue);
    } catch (const std::exception& e) {
        YIO_LOG_ERROR_EVENT("GeolocationConfigProvider.InvalidDeviceConfig", e.what());
        return State::NO_UPDATE;
    }

    if (const auto location = Location::fromJson(jsonLocation); locationFromDeviceConfig_ != location) {
        locationFromDeviceConfig_ = location;
        return State::HAS_UPDATE;
    }

    return State::NO_UPDATE;
}

GeolocationConfigProvider::State GeolocationConfigProvider::onSystemConfig(const std::string& configName, const std::string& jsonConfigValue) {
    if (configName != SYSTEM_CONFIG_NAME) {
        return State::NO_UPDATE;
    }

    Json::Value settings;
    try {
        settings = parseJson(jsonConfigValue);
    } catch (const std::exception& e) {
        YIO_LOG_ERROR_EVENT("GeolocationConfigProvider.InvalidSystemConfig", e.what());
        return State::NO_UPDATE;
    }

    auto state = State::NO_UPDATE;

    if (const auto location = Location::fromJson(settings["location"]); locationFromSystemConfig_ != location) {
        locationFromSystemConfig_ = location;
        state = State::HAS_UPDATE;
    }

    if (const auto timezone = Timezone::fromJson(settings["timezone"]); timezoneFromSystemConfig_ != timezone) {
        timezoneFromSystemConfig_ = timezone;
        state = State::HAS_UPDATE;
    }

    return state;
}

const std::optional<Location>& GeolocationConfigProvider::getLocation() const {
    if (locationFromSystemConfig_.has_value()) {
        return locationFromSystemConfig_;
    }
    return locationFromDeviceConfig_;
}

const std::optional<Timezone>& GeolocationConfigProvider::getTimezone() const {
    return timezoneFromSystemConfig_;
}
