#include "timezone_provider.h"

#include <yandex_io/libs/base/utils.h>
#include <yandex_io/libs/json_utils/json_utils.h>
#include <yandex_io/libs/logging/logging.h>

#include <sstream>

YIO_DEFINE_LOG_MODULE("geolocation");

using namespace quasar;
using namespace YandexIO;

TimezoneProvider::TimezoneProvider(std::shared_ptr<IHttpClient> httpClient, const std::string& quasarBackendUrl)
    : httpClient_(std::move(httpClient))
    , quasarBackendUrl_(quasarBackendUrl)
{
}

std::optional<Timezone> TimezoneProvider::request(const Location& location) const {
    std::optional<Timezone> timezone;

    try {
        const auto response = httpClient_->get("get-timezone", buildBackendUrl(location));

        if (!isSuccessHttpCode(response.responseCode)) {
            std::stringstream ss;
            ss << "Request timezone: Http connection error. Response code: " << response.responseCode << ". Body: "
               << response.body;
            throw std::runtime_error(ss.str());
        }

        YIO_LOG_INFO("Got timezone from backend: " << response.body);
        const auto responseJson = parseJson(response.body);
        timezone = Timezone({.timezoneName = getString(responseJson, "timezone"),
                             .timezoneOffsetSec = getInt64(responseJson, "offset_sec")});
    } catch (const std::exception& e) {
        YIO_LOG_WARN(e.what());
    }

    return timezone;
}

std::string TimezoneProvider::buildBackendUrl(const Location& location) const {
    std::stringstream ss;
    ss << quasarBackendUrl_ << "/get_timezone?lat=" << location.latitude << "&lon=" << location.longitude;
    return ss.str();
}
