#include "keyboard_manager.h"

#include <yandex_io/libs/json_utils/json_utils.h>

#include <iostream>
#include <map>
#include <string_view>
#include <limits>

namespace {
    constexpr char C_EXIT = 'x';
    constexpr char C_START_CONVERSATION = 's';
    constexpr char C_TOGGLE_CONVERSATION = 't';
    constexpr char C_CANCEL_CONVERSATION = 'c';
    constexpr char C_TOGGLE_PLAY_PAUSE = 'p';
    constexpr char C_PLAYER_NEXT_TRACK = 'n';
    constexpr char C_PLAYER_LAST_TRACK = 'l';
    constexpr char C_PLAYER_PAUSE = 'o';
    constexpr char C_TEXT_INPUT_QUERY = 'q';

    std::map<char, std::string> gKeysHelp{
        {C_EXIT, "Exit..."},
        {C_START_CONVERSATION, "Start Conversation"},
        {C_TOGGLE_CONVERSATION, "Toggle Conversation"},
        {C_CANCEL_CONVERSATION, "Cancel Conversation"},
        {C_TOGGLE_PLAY_PAUSE, "Toggle PlayPause"},
        {C_PLAYER_LAST_TRACK, "Player Prev Track"},
        {C_PLAYER_NEXT_TRACK, "Player Next Track"},
        {C_PLAYER_PAUSE, "Player Pause"},
        {C_TEXT_INPUT_QUERY, "Text Input Query"},
    };

    void printHelp() {
        std::cout << "Commands: \n";
        for (const auto& [key, help] : gKeysHelp) {
            std::cout << '\'' << key << "' " << help << '\n';
        }
        std::cout << std::endl;
    }

} // namespace

KeyboardManager::KeyboardManager(std::shared_ptr<YandexIO::SDKInterface> sdk)
    : sdk_(std::move(sdk))
{
}

void KeyboardManager::run() {
    printHelp(); // introduce commands
    while (true) {
        bool exit = false;
        char c;
        std::cin >> c;
        if (std::cin.eof()) {
            break;
        }
        if (gKeysHelp.count(c) > 0) {
            std::cout << "UserInput: " << gKeysHelp[c] << std::endl;
        } else {
            printHelp();
        }
        switch (c) {
            case C_EXIT: {
                exit = true;
                break;
            }
            case C_TOGGLE_CONVERSATION: {
                sdk_->getAliceCapability()->toggleConversation(YandexIO::VinsRequest::createHardwareButtonClickEventSource());
                break;
            }
            case C_START_CONVERSATION: {
                sdk_->getAliceCapability()->startConversation(YandexIO::VinsRequest::createHardwareButtonClickEventSource());
                break;
            }
            case C_CANCEL_CONVERSATION: {
                sdk_->getAliceCapability()->stopConversation();
                break;
            }
            case C_TOGGLE_PLAY_PAUSE: {
                sdk_->getPlaybackControlCapability()->togglePlayPause(true);
                break;
            }
            case C_PLAYER_NEXT_TRACK: {
                sdk_->getPlaybackControlCapability()->next();
                break;
            }
            case C_PLAYER_LAST_TRACK: {
                sdk_->getPlaybackControlCapability()->prev();
                break;
            }
            case C_PLAYER_PAUSE: {
                sdk_->getPlaybackControlCapability()->pause();
                break;
            }
            case C_TEXT_INPUT_QUERY: {
                std::cin.ignore(std::numeric_limits<std::streamsize>::max(), '\n'); // eat \n and go to query
                std::string query;
                std::getline(std::cin, query);
                if (!query.empty()) {
                    Json::Value event;
                    event["text"] = query;
                    event["type"] = "text_input";

                    auto request = std::make_shared<YandexIO::VinsRequest>(
                        std::move(event),
                        YandexIO::VinsRequest::createSoftwareDirectiveEventSource());

                    sdk_->getAliceCapability()->startRequest(std::move(request), nullptr);
                }
            }
            default: {
                break;
            }
        }
        if (exit) {
            break;
        }
    }
}
