#include "led_capability.h"

#include <yandex_io/libs/protobuf_utils/json.h>
#include <yandex_io/libs/base/directives.h>
#include <yandex_io/libs/json_utils/json_utils.h>
#include <yandex_io/libs/logging/logging.h>

YIO_DEFINE_LOG_MODULE("led_capability");

using namespace YandexIO;

namespace {
    NAlice::TAnimationCapability::TDrawAnimationDirective::ESpeakingAnimationPolicy speakingPolicyFromString(const std::string& policy) {
        if (policy == "PlaySpeakingEndOfTts") {
            return NAlice::TAnimationCapability::TDrawAnimationDirective::PlaySpeakingEndOfTts;
        } else if (policy == "SkipSpeakingAnimation") {
            return NAlice::TAnimationCapability::TDrawAnimationDirective::SkipSpeakingAnimation;
        } else {
            return NAlice::TAnimationCapability::TDrawAnimationDirective::Default;
        }
    }

    NAlice::TAnimationCapability::TDrawAnimationDirective::EAnimationStopPolicy stopPolicyFromString(const std::string& policy) {
        if (policy == "PlayOnce") {
            return NAlice::TAnimationCapability::TDrawAnimationDirective::PlayOnce;
        } else if (policy == "PlayOnceTillEndOfTTS") {
            return NAlice::TAnimationCapability::TDrawAnimationDirective::PlayOnceTillEndOfTTS;
        } else if (policy == "RepeatLastTillEndOfTTS") {
            return NAlice::TAnimationCapability::TDrawAnimationDirective::RepeatLastTillEndOfTTS;
        } else if (policy == "RepeatLastTillNextDirective") {
            return NAlice::TAnimationCapability::TDrawAnimationDirective::RepeatLastTillNextDirective;
        } else {
            return NAlice::TAnimationCapability::TDrawAnimationDirective::Unknown;
        }
    }
} // namespace

const std::string LedCapability::DRAW_ANIMATION_DIRECTIVE = "draw_animation_directive";

LedCapability::ILedAnimationListener::~ILedAnimationListener() = default;

LedCapability::LedCapability(std::weak_ptr<ILedAnimationListener> listener, const std::vector<NAlice::TAnimationCapability::EFormat>& supportedFormats)
    : listener_(std::move(listener))
    , state_(createState(supportedFormats))
{
}

LedCapability::~LedCapability() {
    queue_.destroy();
}

void LedCapability::setUpLocalLoader(std::shared_ptr<IScenarioLocalAnimationLoader> localLoader, std::set<std::string> supportedScenarios) {
    loader_ = std::move(localLoader);
    supportedScenarios_ = std::move(supportedScenarios);
}

void LedCapability::setUpS3Downloader(std::shared_ptr<IAnimationLoader> loader, std::unique_ptr<IFileCache> cache, std::shared_ptr<IDevice> device, const S3Downloader::Params& params) {
    downloader_ = std::make_unique<S3Downloader>(std::move(loader), std::move(cache), std::move(device), params);
}

NAlice::TCapabilityHolder LedCapability::getState() const {
    return state_;
}

NAlice::TCapabilityHolder LedCapability::createState(const std::vector<NAlice::TAnimationCapability::EFormat>& supportedFormats) {
    NAlice::TCapabilityHolder capability;
    auto animationCapability = capability.mutable_animationcapability();

    auto meta = animationCapability->mutable_meta();
    meta->add_supporteddirectives(NAlice::TCapability::DrawAnimationDirectiveType);

    auto parameters = animationCapability->mutable_parameters();
    for (const auto& format : supportedFormats) {
        parameters->add_supportedformats(format);
    }
    return capability;
}

YandexIO::IDirectiveHandlerPtr LedCapability::getDirectiveHandler() {
    return shared_from_this();
}

void LedCapability::addListener(std::weak_ptr<YandexIO::ICapability::IListener> /*wlistener*/) {
    // ¯\_(ツ)_/¯
}

void LedCapability::removeListener(std::weak_ptr<YandexIO::ICapability::IListener> /*wlistener*/) {
    // ¯\_(ツ)_/¯
}

const std::string& LedCapability::getHandlerName() const {
    static const std::string NAME = "LedCapability";
    return NAME;
}

const std::set<std::string>& LedCapability::getSupportedDirectiveNames() const {
    static const std::set<std::string> SUPPORTED_DIRECTIVES = {
        DRAW_ANIMATION_DIRECTIVE,
    };
    return SUPPORTED_DIRECTIVES;
}

void LedCapability::handleDirective(const std::shared_ptr<YandexIO::Directive>& directive) {
    queue_.add([this, directive]() {
        const auto drawDirective = quasar::convertJsonToProtobuf<NAlice::TAnimationCapability::TDrawAnimationDirective>(quasar::jsonToString(directive->getData().payload));
        if (!drawDirective.has_value()) {
            YIO_LOG_ERROR_EVENT("LedCapability.handleDirective", "Unable to parse payload: " << directive->getData().payload);
            return;
        }
        std::vector<std::vector<std::shared_ptr<Animation>>> animationSequence;
        for (const auto& animation : drawDirective->GetAnimations()) {
            if (animation.HasS3Directory()) {
                if (auto downloadedAnimations = handleS3Format(animation.GetS3Directory()); !downloadedAnimations.empty()) {
                    animationSequence.push_back(std::move(downloadedAnimations));
                }
            } else if (animation.HasBinaryAnimation()) {
                // TODO handle binary format
            }
        }
        if (const auto listener = listener_.lock(); listener) {
            listener->onScenarioAnimations(std::move(animationSequence), drawDirective->GetSpeakingAnimationPolicy(), drawDirective->GetAnimationStopPolicy());
        }
    });
}

void LedCapability::cancelDirective(const std::shared_ptr<YandexIO::Directive>& /*directive*/) {
    // ¯\_(ツ)_/¯
}

void LedCapability::prefetchDirective(const std::shared_ptr<YandexIO::Directive>& /*directive*/) {
    // ¯\_(ツ)_/¯
}

void LedCapability::onDirective(const std::string& name, const std::string& /*vinsRequestId*/, const std::string& /*jsonPayload*/) {
    if (!supportedScenarios_.contains(name)) {
        return;
    }
    if (loader_ == nullptr) {
        YIO_LOG_ERROR_EVENT("LedCapability.onDirective", "Failed to load local animations: loader is null.");
        return;
    }
    if (auto animations = loader_->loadScenarioLocalAnimations(name);
        !animations.empty()) {
        if (const auto listener = listener_.lock(); listener) {
            listener->onScenarioAnimations(std::move(animations), speakingPolicyFromString("Default"), stopPolicyFromString("Unknown"));
        }
    } else {
        YIO_LOG_WARN("Got empty list of animations from local load.");
    }
}

std::vector<std::shared_ptr<Animation>> LedCapability::handleS3Format(const NAlice::TAnimationCapability::TDrawAnimationDirective::TAnimation::TS3Directory& s3Payload) {
    if (downloader_ == nullptr) {
        YIO_LOG_ERROR_EVENT("LedCapability.handleDirective", "Failed to download animation from s3: s3downloader is null.");
        return {};
    }
    return downloader_->getAnimations(s3Payload.GetBucket(), s3Payload.GetPath());
}
