#include "persistent_cache.h"

#include <yandex_io/libs/base/utils.h>

#include <algorithm>

using namespace YandexIO;

namespace {
    constexpr size_t MAX_ENTRIES_DEFAULT = 10;
} // namespace

PersistentLruCache::PersistentLruCache(TFsPath cachePath)
    : cachePath_(std::move(cachePath))
    , maxEntries_(MAX_ENTRIES_DEFAULT)
{
    Y_VERIFY(cachePath_);
    cachePath_.MkDirs();
    for (const auto& file : quasar::getDirectoryFileList(cachePath_.c_str())) {
        if (!file.starts_with(".")) {
            cache_.push_back(TString(cachePath_ / file));
        }
    }
    shrinkToFit();
}

std::string PersistentLruCache::filenameFromUrl(const std::string& url) const {
    const auto filename = quasar::urlToFilePath(url);
    return TString(cachePath_ / filename);
}

IFileCache::CacheResult PersistentLruCache::getEntry(const std::string& key) {
    auto filename = filenameFromUrl(key);
    if (const auto it = findInCache(filename); it != cache_.end()) {
        cache_.splice(cache_.begin(), cache_, it);
        return {.filename = std::move(filename), .hit = true};
    } else {
        return {.filename = std::move(filename), .hit = false};
    }
}

int PersistentLruCache::clear() {
    size_t size = cache_.size();
    for (const auto& file : cache_) {
        TFsPath(file).ForceDelete();
    }
    cache_.clear();
    return size;
}

std::string PersistentLruCache::putEntry(const std::string& key) {
    auto filename = filenameFromUrl(key);
    if (const auto it = findInCache(key); it != cache_.end()) {
        cache_.splice(cache_.begin(), cache_, it);
    } else if (maxEntries_ == cache_.size()) {
        TFsPath(cache_.back()).ForceDelete();
        cache_.pop_back();
        cache_.push_front(filename);
    } else {
        cache_.push_front(filename);
    }
    return filename;
}

void PersistentLruCache::removeFile(const std::string& filename) {
    if (const auto it = findInCache(filename); it != cache_.end()) {
        TFsPath(filename).ForceDelete();
        cache_.erase(it);
    }
}

void PersistentLruCache::invalidateEntry(const std::string& key) {
    const auto filename = filenameFromUrl(key);
    removeFile(filename);
}

void PersistentLruCache::setSizeEntries(uint64_t sizeEntries) {
    maxEntries_ = sizeEntries;
    shrinkToFit();
}

void PersistentLruCache::shrinkToFit() {
    while (cache_.size() > maxEntries_) {
        TFsPath(cache_.back()).ForceDelete();
        cache_.pop_back();
    }
}

void PersistentLruCache::setSizeKbytes(int /*sizeKbytes*/) {
}
