#include "s3_downloader.h"

#include <yandex_io/libs/logging/logging.h>
#include <boost/algorithm/string.hpp>

#include <util/system/yassert.h>

using namespace YandexIO;

YIO_DEFINE_LOG_MODULE("led_capability");

S3Downloader::S3Downloader(std::shared_ptr<IAnimationLoader> loader, std::unique_ptr<IFileCache> cache, std::shared_ptr<IDevice> device, const Params& params)
    : directory_(params.directory)
    , metaFile_(params.metaFile)
    , loader_(std::move(loader))
    , cache_(std::move(cache))
    , device_(std::move(device))
    , client_(params.name, device_)
{
    Y_VERIFY(cache_);
    Y_VERIFY(device_);
    Y_VERIFY(loader_);
    client_.setTimeout(params.timeout);
    client_.setConnectionTimeout(params.timeout);
    client_.setRetriesCount(params.retries);
    cache_->setSizeEntries(params.cacheEntries);
    cache_->setSizeKbytes(params.cacheSizeKb);
}

std::vector<std::string> S3Downloader::getAnimationDownloadUrls(const std::string& bucketUrl, const std::string& path) {
    try {
        const auto response = client_.get("get_meta", boost::algorithm::join(std::vector<std::string>{bucketUrl, path, directory_, metaFile_}, "/"));
        if (response.responseCode != 200) {
            YIO_LOG_ERROR_EVENT("S3Downloader.getAnimationDownloadUrls", "List request failed with code " << response.responseCode << " " << response.body);
            return {};
        }
        YIO_LOG_INFO("Files found in s3:\n"
                     << response.body);
        std::stringstream ss(response.body);
        std::string path;
        std::vector<std::string> result;
        while (std::getline(ss, path)) {
            result.push_back(bucketUrl + '/' + path);
        }
        return result;
    } catch (const std::exception& ex) {
        YIO_LOG_ERROR_EVENT("S3Downloader.getAnimationDownloadUrls", "Can't download animation, error: " << ex.what());
    }
    return {};
}

std::vector<std::string> S3Downloader::getAnimationFilenames(const std::vector<std::string>& urls) {
    std::vector<quasar::HttpClient::DownloadFileTask> downloadFileTasksList;
    std::vector<std::string> filenames;
    filenames.reserve(urls.size());
    for (const auto& url : urls) {
        const auto entry = cache_->getEntry(url);
        filenames.push_back(entry.filename);
        if (!entry.hit) {
            YIO_LOG_INFO("File: " << entry.filename << " not found in chache. Download it.");
            downloadFileTasksList.push_back(quasar::HttpClient::DownloadFileTask{.url = url, .outputFileName = entry.filename});
        }
    }
    if (!downloadFileTasksList.empty()) {
        try {
            client_.download("animations", downloadFileTasksList);
        } catch (const std::runtime_error& e) {
            YIO_LOG_ERROR_EVENT("S3Downloader.DownloadAnimationFailed", "Can't download animation, error: " << e.what());
            device_->telemetry()->reportEvent("frontalAnimationDownloadFail", e.what());
            for (const auto& task : downloadFileTasksList) {
                cache_->invalidateEntry(task.url);
            }
            return {};
        }
    }
    for (const auto& task : downloadFileTasksList) {
        cache_->putEntry(task.url);
    }
    return filenames;
}

std::vector<std::shared_ptr<Animation>> S3Downloader::getAnimations(const std::string& bucket, const std::string& path) {
    const auto urls = getAnimationDownloadUrls(bucket, path);
    const auto filenames = getAnimationFilenames(urls);
    std::vector<std::shared_ptr<Animation>> animations;
    animations.reserve(filenames.size());
    for (const auto& filename : filenames) {
        if (auto animation = loader_->loadAnimation(filename); animation != nullptr) {
            animations.push_back(std::move(animation));
        }
    }
    return animations;
}
