#pragma once

#include <list>
#include <sstream>
#include <string>
#include <unordered_map>
#include <utility>

namespace quasar {

    using BackgroundGroupIdT = std::string;

    template <typename PatternType>
    class BackgroundAnimationManager final {
    public:
        explicit BackgroundAnimationManager(PatternType idlePattern);

        PatternType current() const;

        int currentPriority() const;

        bool exist(BackgroundGroupIdT groupId) const;

        void add(PatternType pattern, int priority, BackgroundGroupIdT groupId);

        /**
         * @brief Remove current pattern from list (i.e. it can be expired by Duration)
         */
        void pop();

        void remove(BackgroundGroupIdT groupId);

    private:
        template <typename T>
        struct PlayingPatternCtx final {
            PlayingPatternCtx(BackgroundGroupIdT groupId,
                              T pattern, int priority)
                : groupId(std::move(groupId))
                , pattern(std::move(pattern))
                , priority(priority)
            {
            }

            operator std::string() const {
                std::stringstream ss;
                ss << "group: " << groupId << ",";
                ss << "priority: " << priority;
                return ss.str();
            }

            bool operator<(const PlayingPatternCtx& rhs) const {
                return priority < rhs.priority;
            }

            BackgroundGroupIdT groupId;
            T pattern;
            int priority;
        };

        PlayingPatternCtx<PatternType> idlePattern_;

    private:
        void maxPriorityToBack();
        std::string getData() const;

        std::list<PlayingPatternCtx<PatternType>> patternsList_;
        std::unordered_map<BackgroundGroupIdT, typename std::list<PlayingPatternCtx<PatternType>>::iterator> patternsCache_;
    };

}

/* Include class implementation */
#include "background_animation_manager_impl.h"
