#pragma once

#include "background_animation_manager.h"

#include <algorithm>
#include <sstream>
#include <utility>

namespace quasar {

    template <typename PatternType>
    BackgroundAnimationManager<PatternType>::BackgroundAnimationManager(PatternType idlePattern)
        : idlePattern_(BackgroundGroupIdT(), std::move(idlePattern), 0)
    {
    }

    template <typename PatternType>
    PatternType BackgroundAnimationManager<PatternType>::current() const {
        const auto& current = patternsList_.empty() ? idlePattern_ : patternsList_.back();

        return current.pattern;
    }

    template <typename PatternType>
    int BackgroundAnimationManager<PatternType>::currentPriority() const {
        const auto& current = patternsList_.empty() ? idlePattern_ : patternsList_.back();
        return current.priority;
    }

    template <typename PatternType>
    bool BackgroundAnimationManager<PatternType>::exist(BackgroundGroupIdT groupId) const {
        return patternsCache_.count(groupId) > 0;
    }

    template <typename PatternType>
    void BackgroundAnimationManager<PatternType>::add(PatternType pattern, int priority, BackgroundGroupIdT groupId)
    {
        auto itPattern = patternsCache_.find(groupId);

        /* Pattern for group does exist:
         * - check whether it should be replaced due to higher priority
         * - if should, remove old entry
         */

        if (itPattern != patternsCache_.end()) {
            if (itPattern->second->priority > priority) {
                return;
            }

            remove(groupId);
        }

        /*
         * - push new entry into the list. Push new element to the front.
         *   If new element has the same priority as a current playing, max_element
         *   algorithm will take it first.
         * - store iterator to that item in the cache
         * - rebalance list so the highest priority pattern is on the back
         */

        patternsList_.emplace_front(groupId, std::move(pattern), priority);
        patternsCache_[groupId] = patternsList_.begin();

        maxPriorityToBack();
    }

    template <typename PatternType>
    void BackgroundAnimationManager<PatternType>::remove(BackgroundGroupIdT groupId)
    {
        auto itPattern = patternsCache_.find(groupId);

        if (itPattern != patternsCache_.end()) {
            /* Remove pattern from list and cache */

            patternsList_.erase(itPattern->second);
            patternsCache_.erase(itPattern);

            /* Refresh max priority pattern */

            maxPriorityToBack();
        }
    }

    template <typename PatternType>
    void BackgroundAnimationManager<PatternType>::pop()
    {
        if (patternsList_.empty()) {
            return;
        }
        const auto currentId = patternsList_.back().groupId;
        remove(currentId);
    }

    template <typename PatternType>
    void BackgroundAnimationManager<PatternType>::maxPriorityToBack()
    {
        if (!patternsList_.empty()) {
            /* Find out max element iterator and value */

            auto itMax = std::max_element(patternsList_.begin(), patternsList_.end());
            auto valMax = *itMax;

            /* Invalidate its iterator in cache and push it to the back of the list */

            patternsList_.erase(itMax);
            patternsList_.push_back(valMax);

            /* Iterator have changed - refresh cache */

            patternsCache_[valMax.groupId] = prev(patternsList_.end());
        }
    }

    template <typename PatternType>
    std::string BackgroundAnimationManager<PatternType>::getData() const {
        std::stringstream ss;
        for (const auto& it : patternsCache_) {
            ss << "\t CACHE: group = " << it.first << " : cache = " << std::addressof(*it.second);
        }

        for (const auto& it : patternsList_) {
            ss << "\t DATA: group = " << it.groupId << " : iter_addr = " << std::addressof(it);
        }
        return ss.str();
    }

} /* namespace quasar */
