#include "led_animator.h"

#include "ng/default_animation_conductor.h"

#include <yandex_io/libs/logging/logging.h>
#include <yandex_io/libs/threading/utils.h>

#include <algorithm>
#include <cstdlib>
#include <iostream>
#include <utility>

using namespace quasar;

namespace {
    const std::chrono::milliseconds MAX_WAIT_TIME = std::chrono::seconds(1);
} // namespace

LedAnimator::LedAnimator(std::shared_ptr<LedDevices> ledDevices, std::shared_ptr<IFlusher> flusher)
    : LedAnimator(LedPattern::getIdlePattern(ledDevices->getDefaultDevice()->getLedCount(), ledDevices->getDefaultDevice()),
                  ledDevices->getDevicesList().size(),
                  flusher)
{
}

LedAnimator::LedAnimator(std::shared_ptr<Animation> initialAnimation, int devicesCount, std::shared_ptr<IFlusher> flusher)
    : LedAnimator(std::make_shared<DefaultAnimationConductor>(std::move(initialAnimation)), devicesCount, flusher)
{
}

LedAnimator::LedAnimator(std::shared_ptr<AnimationConductor> initialAnimationConductor, int devicesCount, std::shared_ptr<IFlusher> flusher)
    : stopped_(false)
    , devicesCount_(devicesCount)
    , flusher_(std::move(flusher))
{
    play(std::move(initialAnimationConductor));
    drawLoopThread_ = std::thread(&LedAnimator::drawLoop, this);
}

void LedAnimator::play(std::shared_ptr<AnimationConductor> conductor)
{
    conductor->setOnChangedListener([this]() { waitCondition_.notify_one(); });
    YIO_LOG_DEBUG("play, new conductor "
                  << "(" << conductor->getName() << ")");
    std::lock_guard lock(structureMutex_);
    animationChronology_.playNow(std::move(conductor));
    waitCondition_.notify_one();
}

void LedAnimator::stop() {
    YIO_LOG_TRACE("Stopping foreground animation");
    std::unique_lock lock(structureMutex_);
    animationChronology_.removeForeground();
}

void LedAnimator::drawLoop() noexcept {
    std::vector<std::chrono::time_point<std::chrono::steady_clock>> animationWaitTime(devicesCount_);
    setThreadSchedulingParams(SCHED_RR, 99);
    bool hasNew = false;
    auto waitUntil = std::chrono::steady_clock::now() + MAX_WAIT_TIME;
    while (!stopped_.load())
    {
        std::unique_lock lock(structureMutex_);
        const auto& animations = animationChronology_.getAnimations();
        if (hasNew || animationChronology_.hasNew()) {
            std::fill(animationWaitTime.begin(), animationWaitTime.end(), std::chrono::steady_clock::now());
            for (const auto& animation : animations) {
                if (!animation || animation->finished()) {
                    continue;
                }
                YIO_LOG_DEBUG("seeing new animation, starting it: " << animation->getName());
                animation->startAnimationFrom(std::chrono::steady_clock::now());
            }
        }

        waitUntil = std::chrono::steady_clock::now() + MAX_WAIT_TIME;
        const auto timePointAfterWait = std::chrono::steady_clock::now();

        int i = 0;
        for (const auto& animation : animations) {
            if (!animation || animation->finished()) {
                continue;
            }

            // draw new frame of this animation only if time to show some other previous frame has passed
            if (timePointAfterWait >= animationWaitTime[i]) {
                lock.unlock();
                animation->drawCurrentFrame();
                lock.lock();
            }

            Animation::TimePoint timePoint = animation->getEndOfFrameTimePoint();
            animationWaitTime[i] = timePoint;
            if (waitUntil > animation->getEndOfFrameTimePoint()) {
                waitUntil = animation->getEndOfFrameTimePoint();
            }
            ++i;
        }

        // Flush buffers if flusher exists
        if (flusher_) {
            flusher_->flush();
        }

        waitCondition_.wait_until(lock, waitUntil,
                                  [&]() {
                                      return stopped_.load() ||
                                             animationChronology_.hasNew();
                                  });

        hasNew = animationChronology_.hasNew();
        animationChronology_.updateTime(std::chrono::steady_clock::now());
    }
}

LedAnimator::~LedAnimator()
{
    stopped_.store(true);
    waitCondition_.notify_one();
    drawLoopThread_.join();
}
