#pragma once

#include "ng/animation_chronology.h"

#include <yandex_io/modules/leds/led_controller/led_controller.h>
#include <yandex_io/modules/leds/led_patterns/led_pattern.h>

#include <yandex_io/libs/threading/steady_condition_variable.h>

#include <atomic>
#include <chrono>
#include <list>
#include <mutex>
#include <sstream>
#include <thread>
#include <unordered_map>

namespace quasar {

    class LedAnimator {
    public:
        /**
         * @~russian
         * Этот класс используется для синхронной отрисовки анимаций на разных лед-девайсах.
         * Зачем это может понадобиться: если на устройстве есть два логически разных девайса, которые пишут данные в один физический девайс,
         * то мы можем захотеть писать данные не два раза, а один на цикл.
         * В таком случае реализация drawCurrentFrame() этих девайсов будет просто заполнять буфер, который будет записываться физически только на вызов этого флашера.
         * @~english
         * This class can be used for synchronous drawing on different led-devices.
         * What's the use case: if there are 2 logical devices with the same physical device underlying, we can draw leds once per loop.
         * We just implement drawCurrentFrame() on these devices so they don't actually draw leds but only fill buffer. And then this flusher does actual drawing.
         */
        class IFlusher {
        public:
            virtual void flush() = 0;

            virtual ~IFlusher() = default;
        };

    public:
        LedAnimator(std::shared_ptr<LedDevices> ledDevices, std::shared_ptr<IFlusher> flusher = nullptr);
        LedAnimator(std::shared_ptr<Animation> initialAnimation, int devicesCount = 1, std::shared_ptr<IFlusher> flusher = nullptr);
        LedAnimator(std::shared_ptr<AnimationConductor> initialAnimationConductor, int devicesCount = 1, std::shared_ptr<IFlusher> flusher = nullptr);

        void play(std::shared_ptr<AnimationConductor> conductor);

        // stops foreground animation
        // deprecated legacy. Use AnimationConductor
        void stop();

        ~LedAnimator();

    private:
        void drawLoop() noexcept;

        std::thread drawLoopThread_;

        std::atomic<bool> stopped_;

        quasar::SteadyConditionVariable waitCondition_;
        std::mutex structureMutex_;

        AnimationChronology animationChronology_;

        const int devicesCount_;

        std::shared_ptr<IFlusher> flusher_;

        LedAnimator(const LedAnimator& other) = delete;
        LedAnimator& operator=(const LedAnimator& other) = delete;
    };

} // namespace quasar
