#include "led_manager.h"

#include <yandex_io/modules/leds/led_patterns/volume_pattern.h>

#include <yandex_io/libs/device/device.h>
#include <yandex_io/libs/errno/errno_exception.h>
#include <yandex_io/libs/json_utils/json_utils.h>
#include <yandex_io/libs/logging/logging.h>

#include <util/generic/scope.h>
#include <util/system/yassert.h>

#include <dirent.h>

YIO_DEFINE_LOG_MODULE("leds");

using namespace quasar;

LedManager::LedManager(std::shared_ptr<LedDevices> ledDevices, const std::string& ledPatternsPath)
    : LedManager(ledDevices, ledPatternsPath, std::make_unique<LedAnimator>(ledDevices))
{
}

LedManager::LedManager(std::shared_ptr<LedDevices> ledDevices, const std::string& ledPatternsPath, std::unique_ptr<LedAnimator> animator)
    : ledDevices_(std::move(ledDevices))
    , animator_(std::move(animator))
{
    Y_VERIFY(ledDevices_.get() != nullptr);
    Y_VERIFY(!ledPatternsPath.empty());

    loadPatterns(ledPatternsPath);
}

void LedManager::play(std::shared_ptr<AnimationConductor> conductor) {
    animator_->play(std::move(conductor));
}

void LedManager::stop() {
    animator_->stop();
}

void LedManager::loadPatterns(const std::string& pathToPatternsDirectory)
{
    DIR* dp = opendir(pathToPatternsDirectory.c_str());

    if (dp == nullptr) {
        YIO_LOG_ERROR_EVENT("LedManager.LoadPatterns.OpenDirFailed", "Cannot open directory " + pathToPatternsDirectory);
        throw std::runtime_error("Cannot open directory " + pathToPatternsDirectory);
    }

    Y_DEFER {
        closedir(dp);
    };

    dirent* ep = nullptr;
    while ((ep = readdir(dp)) != nullptr) {
        const char* file = ep->d_name;
        if (strstr(file, ".led") && !strstr(file, ".led~")) {
            std::string pathToPattern = pathToPatternsDirectory;
            if (pathToPattern.back() != '/') {
                pathToPattern += '/';
            }
            pathToPattern += file;
            try {
                patterns_[std::string(file)] = LedPattern::loadFromFile(pathToPattern, ledDevices_->getDefaultDevice());
            } catch (const std::exception& e) {
                YIO_LOG_WARN("Cannot load pattern from file " << pathToPattern << ": " << e.what());
            }
        }
    }
}

std::shared_ptr<LedPattern> LedManager::getPattern(const std::string& name) const {
    if (patterns_.count(name) != 0) {
        auto pattern = std::make_shared<LedPattern>(*patterns_.at(name));
        pattern->resetAnimation();
        return pattern;
    } else {
        YIO_LOG_ERROR_EVENT("LedManager.UnexpectedPattern", "Cannot find pattern " + name);
        throw std::runtime_error("Cannot find pattern " + name);
    }
}

std::shared_ptr<VolumePattern> LedManager::getVolumePattern(double volume, bool looped) const {
    auto baseVolumePattern = getPattern("volume.led");
    return std::make_shared<VolumePattern>(std::move(baseVolumePattern), volume, looped);
}

std::shared_ptr<ListeningPattern> LedManager::getListeningPattern(double doaAngle) const {
    const bool invertRotation = false;
    return std::make_shared<ListeningPattern>(getPattern("activate-alisa.led"), doaAngle, invertRotation);
}

std::shared_ptr<ProgressPattern> LedManager::getProgressPattern() const {
    return std::make_shared<ProgressPattern>(getPattern("progress.led"));
}

std::shared_ptr<LedPattern> LedManager::getIdlePattern() const {
    return LedPattern::getIdlePattern(ledDevices_->getDefaultDevice()->getLedCount(), ledDevices_->getDefaultDevice());
}
