#pragma once

#include "led_devices.h"

#include <yandex_io/modules/leds/led_controller/led_device.h>

#include <chrono>
#include <functional>
#include <memory>
#include <string>

/**
 * Animation on single led device.
 * For multi-device animation @see AnimationComposition
 *
 * Single-threaded, i.e. non-thread-safe unless specified otherwise.
 * the basic protocol is as follows:
 *
 * startAnimationFrom(now()) //any number of times
 *
 * in loop:
 *
 *  drawCurrentFrame()
 *  sleepTo(getEndOfFrameTimePoint())
 *  updateTime(now())
 *  if finished()
 *      exit
 *
 * Complex logic, expensive computations, never mind disk access
 * MUST NOT reside inside any of Animation methods. Use separate thread for that.
 *
 * getLength() is permitted to be slow, but should never be called inside draw loop.
 *
 */
class Animation {
public:
    using TimePoint = std::chrono::time_point<std::chrono::steady_clock, std::chrono::nanoseconds>;

    explicit Animation(std::weak_ptr<quasar::LedDevice> ledDevice);
    virtual ~Animation() = default;

    std::weak_ptr<quasar::LedDevice> getDevice();
    std::function<void()> reloadCallback;

    /**
     * Do actual drawing work
     */
    virtual void drawCurrentFrame() = 0;

    /**
     * Move time forward to point.
     *
     * This should be used to update frame of this animation to match current time.
     * In any case, there is no obligation to support moving time backward.
     * steady_clock guarantees that it will not
     *
     * @param timePoint to move to
     */
    virtual void updateTime(TimePoint timePoint) = 0;

    /**
     * Start animation from given time
     * If animation was already started, MUST do nothing
     * @param timePoint to start from
     */
    virtual void startAnimationFrom(TimePoint timePoint) = 0;

    /**
     * Indicator if animation had run out of frames to draw or time to show it.
     *
     * @returns true if animation is done and it is probably time to draw something else.
     *
     * If Animation.drawCurrentFrame should no longer be called, finished() must return true
     *
     * If you override finished(), consider also overriding updateTime, startAnimationFrom, as they should work in concert
     */
    [[nodiscard]] virtual bool finished() const = 0;

    /**
     * getEndOfFrameTimePoint() is used to determine frame rate of animation.
     * drawCurrentFrame will be called no more than once until steady_clock will reach value returned by getEndOfFrameTimePoint()
     *
     * @return timePoint until which to sleep
     */
    [[nodiscard]] virtual TimePoint getEndOfFrameTimePoint() const = 0;
    [[nodiscard]] virtual std::string getName() const;

    /**
     * Length of animation
     *
     * N.B. O(number of frames), so use wisely
     *
     * @return total length of animation
     */
    [[nodiscard]] virtual std::chrono::nanoseconds getLength() const = 0;

    /**
     * start animation over, i.e. from frame 0
     */
    virtual void resetAnimation() = 0;

protected:
    std::weak_ptr<quasar::LedDevice> ledDevice_;
};
